import React, { useEffect, useState } from "react";
import {
  Box,
  Button,
  Heading,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  useDisclosure,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalCloseButton,
  Image,
  IconButton,
  HStack,
  useToast,
} from "@chakra-ui/react";
import { AddIcon, ArrowBackIcon, EditIcon, ViewIcon, DeleteIcon } from "@chakra-ui/icons";
import { useNavigate } from "react-router-dom";
import axios from "../utils/axios";
import VehiclePickupForm from "../components/VehiclePickupForm";
import VehicleReturnForm from "../components/VehicleReturnForm";
import VehicleEditForm from "../components/VehicleEditForm";
import { useAuth } from "../context/AuthContext";
import { formatDate } from "../utils/formatDate";

const VehicleLogbookPage = () => {
  const { user } = useAuth();
  const userId = user?.id;
  const isManager = user?.role === "admin" || user?.role === "branch_admin";

  const [logbookEntries, setLogbookEntries] = useState([]);
  const [vehicles, setVehicles] = useState([]);
  const [selectedEntry, setSelectedEntry] = useState(null);
  const [selectedSignature, setSelectedSignature] = useState(null);
  const [editMode, setEditMode] = useState("pickup");

  const { isOpen, onOpen, onClose } = useDisclosure();
  const {
    isOpen: isReturnOpen,
    onOpen: onOpenReturnModal,
    onClose: onCloseReturnModal,
  } = useDisclosure();
  const {
    isOpen: isSignatureOpen,
    onOpen: onOpenSignatureModal,
    onClose: onCloseSignatureModal,
  } = useDisclosure();
  const {
    isOpen: isEditOpen,
    onOpen: onOpenEditModal,
    onClose: onCloseEditModal,
  } = useDisclosure();

  const navigate = useNavigate();
  const toast = useToast();

  const fetchLogbook = async () => {
    const res = await axios.get("/vehicle-logbook");
    setLogbookEntries(res.data);
  };

  const fetchVehicles = async () => {
    const res = await axios.get("/vehicles");
    setVehicles(res.data);
  };

  useEffect(() => {
    fetchLogbook();
    fetchVehicles();
  }, []);

  const handleShowSignature = async (logbookId, type) => {
    try {
      const res = await axios.get(`/vehicle-logbook/${logbookId}/signature?type=${type}`);
      setSelectedSignature(res.data.url);
      onOpenSignatureModal();
    } catch (err) {
      console.error(err);
      alert("Σφάλμα κατά την προβολή υπογραφής.");
    }
  };

  const handleDelete = async (id) => {
    if (!window.confirm("Είστε σίγουροι ότι θέλετε να διαγράψετε την εγγραφή;")) return;
    try {
      await axios.delete(`/vehicle-logbook/${id}`);
      toast({ title: "Η εγγραφή διαγράφηκε.", status: "success", duration: 3000 });
      fetchLogbook();
    } catch (err) {
      console.error(err);
      toast({ title: "Αποτυχία διαγραφής.", status: "error", duration: 3000 });
    }
  };

  return (
    <Box p={4} maxW="100vw">
      <Box display="flex" flexWrap="wrap" justifyContent="space-between" alignItems="center" mb={4} gap={2}>
        <Heading size={{ base: "md", md: "lg" }}>Βιβλίο Κίνησης Οχημάτων</Heading>
        <Button leftIcon={<ArrowBackIcon />} onClick={() => navigate(-1)} colorScheme="gray">
          Πίσω
        </Button>
      </Box>

      <Button
        colorScheme="blue"
        mb={4}
        onClick={onOpen}
        leftIcon={<AddIcon />}
        width={{ base: "100%", sm: "auto" }}
      >
        Νέα Παραλαβή
      </Button>

      <Box overflowX="auto">
        <Table variant="simple" size="sm">
          <Thead>
            <Tr>
              <Th>Όχημα</Th>
              <Th>Οδηγός</Th>
              <Th>Έναρξη</Th>
              <Th>Τέλος</Th>
              <Th>Συμβάν;</Th>
              <Th>Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {logbookEntries.map((entry) => (
              <Tr key={entry.id}>
                <Td>{entry.plate}</Td>
                <Td>{entry.username}</Td>
                <Td>{formatDate(entry.pickup_time)}</Td>
<Td>{entry.return_time ? formatDate(entry.return_time) : "—"}</Td>

                <Td>{entry.has_incident ? "Ναι" : "Όχι"}</Td>
                <Td>
                  <HStack spacing={1} flexWrap="wrap">
                    {!entry.return_time && (
                      <Button
                        size="sm"
                        colorScheme="green"
                        onClick={() => {
                          setSelectedEntry(entry);
                          onOpenReturnModal();
                        }}
                      >
                        Επιστροφή
                      </Button>
                    )}

                    {isManager && (
                      <>
                        <IconButton
                          size="sm"
                          icon={<ViewIcon />}
                          aria-label="Υπογραφή"
                          onClick={() => handleShowSignature(entry.id, entry.return_time ? "return" : "pickup")}
                        />

                        <IconButton
                          size="sm"
                          icon={<EditIcon />}
                          aria-label="Τροποποίηση"
                          onClick={() => {
                            setSelectedEntry(entry);
                            setEditMode(entry.return_time ? "return" : "pickup");
                            onOpenEditModal();
                          }}
                        />

                        <IconButton
                          size="sm"
                          icon={<DeleteIcon />}
                          aria-label="Διαγραφή"
                          colorScheme="red"
                          onClick={() => handleDelete(entry.id)}
                        />
                      </>
                    )}
                  </HStack>
                </Td>
              </Tr>
            ))}
          </Tbody>
        </Table>
      </Box>

      {/* Modal: Νέα παραλαβή */}
      <Modal isOpen={isOpen} onClose={onClose} size="xl">
        <ModalOverlay />
        <ModalContent>
          <ModalHeader>Νέα Παραλαβή Οχήματος</ModalHeader>
          <ModalCloseButton />
          <ModalBody>
            <VehiclePickupForm
              vehicles={vehicles}
              userId={userId}
              onSuccess={() => {
                onClose();
                fetchLogbook();
              }}
            />
          </ModalBody>
        </ModalContent>
      </Modal>

      {/* Modal: Επιστροφή */}
      <Modal isOpen={isReturnOpen} onClose={onCloseReturnModal} size="xl">
        <ModalOverlay />
        <ModalContent>
          <ModalHeader>Επιστροφή Οχήματος</ModalHeader>
          <ModalCloseButton />
          <ModalBody>
            {selectedEntry && (
              <VehicleReturnForm
                logbookId={selectedEntry.id}
                onSuccess={() => {
                  onCloseReturnModal();
                  fetchLogbook();
                }}
              />
            )}
          </ModalBody>
        </ModalContent>
      </Modal>

      {/* Modal: Υπογραφή */}
      <Modal isOpen={isSignatureOpen} onClose={onCloseSignatureModal} size="md">
        <ModalOverlay />
        <ModalContent>
          <ModalHeader>Προβολή Υπογραφής</ModalHeader>
          <ModalCloseButton />
          <ModalBody>
            {selectedSignature ? (
              <Image
                src={`http://localhost:5000${selectedSignature}`}
                alt="Υπογραφή"
                borderRadius="md"
                maxW="100%"
              />
            ) : (
              <p>Δεν βρέθηκε υπογραφή.</p>
            )}
          </ModalBody>
        </ModalContent>
      </Modal>

      {/* Modal: Τροποποίηση */}
      <Modal isOpen={isEditOpen} onClose={onCloseEditModal} size="xl">
        <ModalOverlay />
        <ModalContent>
          <ModalHeader>
            Τροποποίηση {editMode === "pickup" ? "Παραλαβής" : "Επιστροφής"}
          </ModalHeader>
          <ModalCloseButton />
          <ModalBody>
            {selectedEntry && (
              <VehicleEditForm
                entry={selectedEntry}
                mode={editMode}
                onSuccess={() => {
                  fetchLogbook();
                  onCloseEditModal();
                }}
              />
            )}
          </ModalBody>
        </ModalContent>
      </Modal>
    </Box>
  );
};

export default VehicleLogbookPage;
