import {
  Box,
  Button,
  Heading,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  IconButton,
  useDisclosure,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalCloseButton,
  Input,
  Select,
  VStack,
  HStack,
  useToast,
} from "@chakra-ui/react";
import { AddIcon, EditIcon, DeleteIcon, ArrowBackIcon } from "@chakra-ui/icons";
import { useEffect, useState } from "react";
import axios from "../utils/axios";
import { useNavigate } from "react-router-dom";

interface User {
  id: number;
  username: string;
  role: string;
  branch_id: number | null;
  created_at: string;
}

interface Branch {
  id: number;
  name: string;
}

export default function Users() {
  const [users, setUsers] = useState<User[]>([]);
  const [branches, setBranches] = useState<Branch[]>([]);
  const [form, setForm] = useState<Partial<User>>({});
  const [editingId, setEditingId] = useState<number | null>(null);
  const { isOpen, onOpen, onClose } = useDisclosure();
  const toast = useToast();
  const navigate = useNavigate();

  const fetchUsers = async () => {
    try {
      const res = await axios.get("/users");
      setUsers(res.data);
    } catch (err) {
      console.error("❌ Failed to fetch users", err);
    }
  };

  const fetchBranches = async () => {
    try {
      const res = await axios.get("/branches");
      setBranches(res.data);
    } catch (err) {
      console.error("❌ Failed to fetch branches", err);
    }
  };

  useEffect(() => {
    fetchUsers();
    fetchBranches();
  }, []);

  const handleSave = async () => {
    try {
      if (editingId) {
        await axios.put(`/users/${editingId}`, form);
        toast({ title: "Ο χρήστης ενημερώθηκε", status: "success" });
      } else {
        await axios.post("/users", form);
        toast({ title: "Ο χρήστης δημιουργήθηκε", status: "success" });
      }
      fetchUsers();
      onClose();
      setForm({});
      setEditingId(null);
    } catch (err) {
      console.error("❌ Failed to save user", err);
      toast({ title: "Σφάλμα αποθήκευσης", status: "error" });
    }
  };

  const handleEdit = (user: User) => {
    setForm(user);
    setEditingId(user.id);
    onOpen();
  };

  const handleDelete = async (id: number) => {
    try {
      await axios.delete(`/users/${id}`);
      toast({ title: "Ο χρήστης διαγράφηκε", status: "info" });
      fetchUsers();
    } catch (err) {
      console.error("❌ Failed to delete user", err);
    }
  };

  return (
    <Box p={4} maxW="100vw">
      <HStack justify="space-between" mb={4} flexWrap="wrap" gap={2}>
        <Heading size={{ base: "md", md: "lg" }}>Χρήστες</Heading>
        <Button leftIcon={<ArrowBackIcon />} onClick={() => navigate(-1)}>
          Πίσω
        </Button>
      </HStack>

      <Button
        colorScheme="green"
        leftIcon={<AddIcon />}
        mb={4}
        onClick={onOpen}
        width={{ base: "100%", sm: "auto" }}
      >
        Νέος Χρήστης
      </Button>

      <Box overflowX="auto">
        <Table variant="simple" size="sm">
          <Thead>
            <Tr>
              <Th>Όνομα Χρήστη</Th>
              <Th>Ρόλος</Th>
              <Th>Υποκατάστημα</Th>
              <Th>Ημ/νία Δημιουργίας</Th>
              <Th>Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {users.map((u) => (
              <Tr key={u.id}>
                <Td>{u.username}</Td>
                <Td>{u.role}</Td>
                <Td>
                  {u.branch_id === null
                    ? "Διοίκηση"
                    : branches.find((b) => b.id === u.branch_id)?.name || "?"}
                </Td>
                <Td>{new Date(u.created_at).toLocaleDateString("el-GR")}</Td>
                <Td>
                  <IconButton
                    icon={<EditIcon />}
                    size="sm"
                    mr={2}
                    onClick={() => handleEdit(u)}
                    aria-label="edit"
                  />
                  <IconButton
                    icon={<DeleteIcon />}
                    size="sm"
                    colorScheme="red"
                    onClick={() => handleDelete(u.id)}
                    aria-label="delete"
                  />
                </Td>
              </Tr>
            ))}
          </Tbody>
        </Table>
      </Box>

      <Modal isOpen={isOpen} onClose={onClose} isCentered size={{ base: "full", sm: "md" }}>
        <ModalOverlay />
        <ModalContent>
          <ModalHeader textAlign="center">{editingId ? "Επεξεργασία" : "Νέος Χρήστης"}</ModalHeader>
          <ModalCloseButton />
          <ModalBody>
            <VStack spacing={3} py={4}>
              <Input
                placeholder="Όνομα χρήστη"
                value={form.username || ""}
                onChange={(e) => setForm({ ...form, username: e.target.value })}
                width="100%"
              />
              <Input
                placeholder="Κωδικός"
                type="password"
                onChange={(e) => setForm({ ...form, password: e.target.value })}
                width="100%"
              />
              <Select
                placeholder="Ρόλος"
                value={form.role || ""}
                onChange={(e) => setForm({ ...form, role: e.target.value })}
                width="100%"
              >
                <option value="admin">Admin</option>
                <option value="branch_admin">Υπεύθυνος</option>
                <option value="driver">Οδηγός</option>
                <option value="accountant">Λογιστήριο</option>
                <option value="customer">Πελάτης</option>
              </Select>
              <Select
                placeholder="Υποκατάστημα"
                value={form.branch_id ?? ""}
                onChange={(e) =>
                  setForm({
                    ...form,
                    branch_id: e.target.value ? parseInt(e.target.value) : null,
                  })
                }
                width="100%"
              >
                <option value="">Διοίκηση</option>
                {branches.map((b) => (
                  <option key={b.id} value={b.id}>
                    {b.name}
                  </option>
                ))}
              </Select>
              <Button colorScheme="blue" onClick={handleSave} width="100%">
                Αποθήκευση
              </Button>
            </VStack>
          </ModalBody>
        </ModalContent>
      </Modal>
    </Box>
  );
}
