import {
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalCloseButton,
  ModalBody,
  ModalFooter,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Button,
  Select,
  Textarea,
  HStack,
  VStack,
  useToast,
  Text,
  Spinner,
} from "@chakra-ui/react";
import { useEffect, useState } from "react";
import api from "../utils/axios";
import DatePicker, { registerLocale } from "react-datepicker";
import { el } from "date-fns/locale";
import "react-datepicker/dist/react-datepicker.css";

registerLocale("el", el);

type Payroll = {
  id: number;
  employee_id?: number;
  period?: string;
  year?: number;
};

type Item = {
  id: number;
  payroll_id: number;
  type: string;
  amount_due: number;
  amount_paid?: number;
  remaining?: number;
};

type Payment = {
  id: number;
  payment_date: string;
  amount: number;
  method?: "bank" | "cash" | "card" | "other";
  notes?: string | null;
};

interface Props {
  payroll: Payroll;
  item: Item;
  onClose: () => void;
  refresh?: () => void;
}

export default function PayrollPayments({ payroll, item, onClose, refresh }: Props) {
  const toast = useToast();
  const [payments, setPayments] = useState<Payment[]>([]);
  const [loading, setLoading] = useState<boolean>(true);

  const [amountStr, setAmountStr] = useState<string>("");
  const [paymentDate, setPaymentDate] = useState<string>(() => {
    const d = new Date();
    const yyyy = d.getFullYear();
    const mm = String(d.getMonth() + 1).padStart(2, "0");
    const dd = String(d.getDate()).padStart(2, "0");
    return `${yyyy}-${mm}-${dd}`;
  });
  const [method, setMethod] = useState<string>("bank");
  const [notes, setNotes] = useState<string>("");

  // === Φόρτωση Πληρωμών ===
  const fetchPayments = async () => {
    if (!item?.id) return;
    setLoading(true);
    try {
      const res = await api.get(`/payrolls/items/${item.id}/payments`);
      setPayments(res.data || []);
    } catch (err) {
      console.error("❌ Error fetching payments:", err);
      toast({ title: "Σφάλμα φόρτωσης πληρωμών", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchPayments();
  }, [item?.id]);

  // === Καταχώριση νέας πληρωμής ===
  const handleAddPayment = async () => {
    const parsedAmount = parseFloat((amountStr || "").toString().replace(",", "."));
    if (!parsedAmount || parsedAmount <= 0) {
      toast({ title: "Δώσε έγκυρο ποσό", status: "warning" });
      return; 
    }
    if (!paymentDate) {
      toast({ title: "Διάλεξε ημερομηνία", status: "warning" });
      return;
    }

    try {
      await api.post(`/payrolls/payments`, {
        employee_id: payroll?.employee_id,
        payroll_id: payroll?.id,
        item_id: item?.id,
        amount: parsedAmount,
        payment_date: paymentDate, // ISO format για backend
        method,
        notes,
      });

      toast({ title: "Η πληρωμή καταχωρήθηκε", status: "success" });
      setAmountStr("");
      setNotes("");
      await fetchPayments();
      if (typeof refresh === "function") refresh();
      window.dispatchEvent(new Event("payrolls-updated")); // refresh Dashboard
    } catch (err) {
      console.error("❌ Error adding payment:", err);
      toast({ title: "Αποτυχία καταχώρισης πληρωμής", status: "error" });
    }
  };

  // === Διαγραφή πληρωμής ===
  const handleDeletePayment = async (paymentId: number) => {
    try {
      await api.delete(`/payrolls/payments/${paymentId}`);
      toast({ title: "Η πληρωμή διαγράφηκε", status: "info" });
      await fetchPayments();
      if (typeof refresh === "function") refresh();
      window.dispatchEvent(new Event("payrolls-updated"));
    } catch (err) {
      console.error("❌ Error deleting payment:", err);
      toast({ title: "Σφάλμα διαγραφής πληρωμής", status: "error" });
    }
  };

  return (
    <Modal isOpen={true} onClose={onClose} size="lg">
      <ModalOverlay />
      <ModalContent>
        <ModalHeader>
          Πληρωμές — {item?.type?.toUpperCase()} (Item #{item?.id})
        </ModalHeader>
        <ModalCloseButton />
        <ModalBody>
          <VStack align="stretch" spacing={4}>
            {/* === Νέα Πληρωμή === */}
            <HStack align="flex-end">
              {/* Ποσό */}
              <VStack align="stretch" spacing={1} flex={1}>
                <Text fontSize="sm" color="gray.600">
                  Ποσό (€)
                </Text>
                <input
                  type="number"
                  step="0.01"
                  value={amountStr}
                  onChange={(e) => setAmountStr(e.target.value)}
                  placeholder="π.χ. 150.00"
                  className="chakra-input"
                />
              </VStack>

              {/* Ημερομηνία */}
              <VStack align="stretch" spacing={1} flex={1}>
                <Text fontSize="sm" color="gray.600">
                  Ημερομηνία
                </Text>
                <DatePicker
                  selected={paymentDate ? new Date(paymentDate) : new Date()}
                  onChange={(date: Date | null) => {
                    if (!date) return;
                    const iso = date.toISOString().split("T")[0];
                    setPaymentDate(iso);
                  }}
                  locale="el"
                  dateFormat="dd/MM/yyyy"
                  className="chakra-input css-datepicker"
                  placeholderText="Επιλέξτε ημερομηνία"
                  showPopperArrow={false}
                />
                
              </VStack>

              {/* Μέθοδος */}
              <VStack align="stretch" spacing={1} flex={1}>
                <Text fontSize="sm" color="gray.600">
                  Τρόπος
                </Text>
                <Select value={method} onChange={(e) => setMethod(e.target.value)}>
                  <option value="bank">Κατάθεση</option>
                  <option value="cash">Μετρητά</option>
                  <option value="card">Κάρτα</option>
                  <option value="other">Άλλο</option>
                </Select>
              </VStack>
            </HStack>

            {/* Σημειώσεις */}
            <VStack align="stretch" spacing={1}>
              <Text fontSize="sm" color="gray.600">
                Σημειώσεις
              </Text>
              <Textarea
                value={notes}
                onChange={(e) => setNotes(e.target.value)}
                placeholder="Προαιρετικές σημειώσεις"
              />
            </VStack>

            {/* Κουμπί Καταχώρισης */}
            <HStack justify="flex-end">
              <Button onClick={handleAddPayment} colorScheme="green">
                💾 Καταχώριση Πληρωμής
              </Button>
            </HStack>

            {/* === Ιστορικό Πληρωμών === */}
            <VStack align="stretch" spacing={2}>
              <Text fontWeight="bold">Ιστορικό Πληρωμών</Text>
              {loading ? (
                <Spinner />
              ) : payments.length === 0 ? (
                <Text color="gray.500">Δεν υπάρχουν πληρωμές για το συγκεκριμένο item.</Text>
              ) : (
                <Table size="sm">
                  <Thead bg="gray.100">
                    <Tr>
                      <Th>Ημερομηνία</Th>
                      <Th isNumeric>Ποσό (€)</Th>
                      <Th>Μέθοδος</Th>
                      <Th>Σημειώσεις</Th>
                      <Th>Ενέργειες</Th>
                    </Tr>
                  </Thead>
                  <Tbody>
                    {payments.map((p) => (
                      <Tr key={p.id}>
                        <Td>
                          {new Date(p.payment_date).toLocaleDateString("el-GR", {
                            day: "2-digit",
                            month: "long",
                            year: "numeric",
                          }).toUpperCase()}
                        </Td>
                        <Td isNumeric>{Number(p.amount).toFixed(2)}</Td>
                        <Td>{p.method || "-"}</Td>
                        <Td>{p.notes || "-"}</Td>
                        <Td>
                          <Button
                            size="xs"
                            colorScheme="red"
                            onClick={() => handleDeletePayment(p.id)}
                          >
                            ❌ Διαγραφή
                          </Button>
                        </Td>
                      </Tr>
                    ))}
                  </Tbody>
                </Table>
              )}
            </VStack>
          </VStack>
        </ModalBody>
        <ModalFooter>
          <Button variant="outline" onClick={onClose}>
            Κλείσιμο
          </Button>
        </ModalFooter>
      </ModalContent>
    </Modal>
  );
}
