import {
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalCloseButton,
  ModalBody,
  Table,
  Thead,
  Tr,
  Th,
  Tbody,
  Td,
  Tfoot,
  Button,
  useToast,
  Input,
  HStack,
  VStack,
  Text,
  Select,
} from "@chakra-ui/react";
import { useEffect, useState } from "react";
import api from "../utils/axios";
import PayrollPayments from "./PayrollPayments";

// --- Αντιστοίχιση ENUM values με ελληνικά labels ---
const TYPE_LABELS: Record<string, string> = {
  salary: "Μισθός",
  food_card: "Κάρτα Σίτισης",
  responsibility_allowance: "Επίδομα Ευθύνης",
  overtime: "Υπερωρίες",
  bonus: "Bonus / Επίδομα",
  other: "Άλλο",
};

export default function PayrollItemsModal({ payroll, onClose }: any) {
  const [items, setItems] = useState<any[]>([]);
  const [selectedItem, setSelectedItem] = useState<any | null>(null);
  const [newItem, setNewItem] = useState({ type: "", amount_due: "" });
  const toast = useToast();

  // === Φόρτωση Items ===
  const fetchItems = async () => {
    if (!payroll?.id) return;
    try {
      const res = await api.get(`/payrolls/${payroll.id}/items`);
      setItems(res.data || []);
    } catch (err) {
      console.error("❌ Error fetching payroll items:", err);
      toast({ title: "Σφάλμα φόρτωσης.", status: "error" });
    }
  };

  useEffect(() => {
    fetchItems();
  }, [payroll]);

  // === Προσθήκη Item ===
  const handleAddItem = async () => {
    const parsedAmount = parseFloat(newItem.amount_due.toString().replace(",", "."));
    if (!newItem.type || parsedAmount <= 0) {
      toast({ title: "Συμπλήρωσε έγκυρα πεδία", status: "warning" });
      return;
    }

    try {
      await api.post(`/payrolls/${payroll.id}/items`, {
        type: newItem.type,
        amount_due: parsedAmount,
      });

      toast({ title: "Το item προστέθηκε", status: "success" });
      setNewItem({ type: "", amount_due: "" });
      fetchItems();
    } catch (err) {
      console.error("❌ Error adding item:", err);
      toast({ title: "Αποτυχία προσθήκης", status: "error" });
    }
  };

  // === Διαγραφή Item ===
  const handleDelete = async (id: number) => {
    try {
      await api.delete(`/payrolls/items/${id}`);
fetchItems(); // ✅ ανανεώνει
      setItems((prev) => prev.filter((it) => it.id !== id));
      toast({ title: "Το item διαγράφηκε", status: "info" });
    } catch (err) {
      console.error("❌ Error deleting item:", err);
      toast({ title: "Σφάλμα διαγραφής", status: "error" });
    }
  };

  // === Υπολογισμοί ===
  const totalDue = items.reduce((a, b) => a + Number(b.amount_due || 0), 0);
  const totalPaid = items.reduce((a, b) => a + Number(b.amount_paid || 0), 0);
  const totalRemaining = totalDue - totalPaid;

  return (
    <Modal isOpen={!!payroll} onClose={onClose} size="xl">
      <ModalOverlay />
      <ModalContent>
        <ModalHeader>Ανάλυση Περιόδου {payroll?.period}</ModalHeader>
        <ModalCloseButton />
        <ModalBody>
          <VStack align="stretch" spacing={4}>
            {/* --- Προσθήκη νέου item --- */}
            <HStack>
              <Select
                placeholder="Επιλογή τύπου"
                value={newItem.type}
                onChange={(e) =>
                  setNewItem({ ...newItem, type: e.target.value })
                }
              >
                <option value="salary">Μισθός</option>
                <option value="food_card">Κάρτα Σίτισης</option>
                <option value="responsibility_allowance">Επίδομα Ευθύνης</option>
                <option value="overtime">Υπερωρίες</option>
                <option value="bonus">Bonus / Επίδομα</option>
                <option value="other">Άλλο</option>
              </Select>

              <Input
                type="number"
                step="0.01"
                placeholder="Ποσό €"
                value={newItem.amount_due}
                onChange={(e) =>
                  setNewItem({
                    ...newItem,
                    amount_due: e.target.value,
                  })
                }
              />

              <Button colorScheme="blue" onClick={handleAddItem}>
                ➕ Προσθήκη
              </Button>
            </HStack>

            {/* --- Πίνακας Items --- */}
            {items.length === 0 ? (
              <Text color="gray.500" textAlign="center" py={4}>
                Δεν υπάρχουν εγγραφές για αυτήν την περίοδο.
              </Text>
            ) : (
              <Table size="sm">
                <Thead bg="gray.100">
                  <Tr>
                    <Th>Κατηγορία</Th>
                    <Th isNumeric>Πληρωτέο (€)</Th>
                    <Th isNumeric>Πληρωμένο (€)</Th>
                    <Th isNumeric>Υπόλοιπο (€)</Th>
                    <Th>Κατάσταση</Th>
                    <Th>Ενέργειες</Th>
                  </Tr>
                </Thead>

                <Tbody>
                  {items.map((it) => {
                    const remaining = Number(it.amount_due) - Number(it.amount_paid);
                    return (
                      <Tr key={it.id}>
                        <Td>{TYPE_LABELS[it.type] || it.type}</Td>
                        <Td isNumeric>{Number(it.amount_due).toFixed(2)}</Td>
                        <Td isNumeric>{Number(it.amount_paid).toFixed(2)}</Td>
                        <Td isNumeric>
                          <Text
                            color={remaining > 0 ? "red.500" : "green.600"}
                          >
                            {remaining.toFixed(2)}
                          </Text>
                        </Td>
                        <Td>
                          {remaining <= 0 ? (
                            <Text color="green.500">Εξοφλημένο</Text>
                          ) : Number(it.amount_paid) > 0 ? (
                            <Text color="orange.400">Μερικώς</Text>
                          ) : (
                            <Text color="red.500">Απλήρωτο</Text>
                          )}
                        </Td>
                        <Td>
                          <HStack>
                            <Button
                              size="xs"
                              colorScheme="green"
                              onClick={() => setSelectedItem(it)}
                            >
                              💰 Πληρωμή
                            </Button>
                            <Button
                              size="xs"
                              colorScheme="red"
                              onClick={() => handleDelete(it.id)}
                            >
                              ❌ Διαγραφή
                            </Button>
                          </HStack>
                        </Td>
                      </Tr>
                    );
                  })}
                </Tbody>

                <Tfoot>
                  <Tr fontWeight="bold">
                    <Td>Σύνολα:</Td>
                    <Td isNumeric>{totalDue.toFixed(2)}</Td>
                    <Td isNumeric>{totalPaid.toFixed(2)}</Td>
                    <Td isNumeric>
                      <Text color={totalRemaining > 0 ? "red.500" : "green.600"}>
                        {totalRemaining.toFixed(2)}
                      </Text>
                    </Td>
                    <Td colSpan={2}></Td>
                  </Tr>
                </Tfoot>
              </Table>
            )}
          </VStack>

          {/* --- Modal Πληρωμών --- */}
          {selectedItem && (
            <PayrollPayments
              payroll={payroll}
              item={selectedItem}
              onClose={() => {
                setSelectedItem(null);
                fetchItems(); // ✅ refresh μετά από πληρωμή
              }}
              refresh={fetchItems}
            />
          )}
        </ModalBody>
      </ModalContent>
    </Modal>
  );
}
