import { useEffect, useState } from "react";
import {
  Box,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Button,
  Badge,
  Spinner,
  Text,
  VStack,
  HStack,
  Heading,
  useToast,
  Input,
  Select,
} from "@chakra-ui/react";
import { fetchPayrolls } from "../utils/PayrollAPI";
import PayrollItemsModal from "./PayrollItemsModal";
import { useNavigate } from "react-router-dom";
import api from "../utils/axios";
import ConfirmDialog from "../components/ConfirmDialog";

export default function PayrollDashboard() {
  const [payrolls, setPayrolls] = useState<any[]>([]);
  const [selectedPayroll, setSelectedPayroll] = useState<any | null>(null);
  const [loading, setLoading] = useState(true);
  const [confirmOpen, setConfirmOpen] = useState(false);
  const [payrollToDelete, setPayrollToDelete] = useState<number | null>(null);

  // 🔍 Search filters
  const [searchName, setSearchName] = useState("");
  const [searchPeriod, setSearchPeriod] = useState("");
  const [searchYear, setSearchYear] = useState(new Date().getFullYear().toString());

  const toast = useToast();
  const navigate = useNavigate();

  const greekPeriods = [
    "ΙΑΝΟΥΑΡΙΟΣ",
    "ΦΕΒΡΟΥΑΡΙΟΣ",
    "ΜΑΡΤΙΟΣ",
    "ΑΠΡΙΛΙΟΣ",
    "ΜΑΙΟΣ",
    "ΙΟΥΝΙΟΣ",
    "ΙΟΥΛΙΟΣ",
    "ΑΥΓΟΥΣΤΟΣ",
    "ΣΕΠΤΕΜΒΡΙΟΣ",
    "ΟΚΤΩΒΡΙΟΣ",
    "ΝΟΕΜΒΡΙΟΣ",
    "ΔΕΚΕΜΒΡΙΟΣ",
    "ΔΩΡΟ ΠΑΣΧΑ",
    "ΕΠΙΔΟΜΑ ΑΔΕΙΑΣ",
    "ΔΩΡΟ ΧΡΙΣΤΟΥΓΕΝΝΩΝ",
  ];

  // === Φόρτωση μισθοδοσιών ===
  const loadPayrolls = async () => {
    try {
      setLoading(true);
      const data = await fetchPayrolls(
        Number(searchYear),
        searchName.trim(),
        searchPeriod.trim()
      );
      setPayrolls(data);
    } catch (err) {
      console.error("❌ Error fetching payrolls:", err);
      toast({
        title: "Σφάλμα φόρτωσης μισθοδοσιών",
        description: "Δεν βρέθηκαν δεδομένα ή υπάρχει πρόβλημα με τον server.",
        status: "error",
      });
      setPayrolls([]);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    loadPayrolls();
  }, []);

  // === Event listener για refresh από άλλα components ===
  useEffect(() => {
    const refreshHandler = () => loadPayrolls();
    window.addEventListener("payrolls-updated", refreshHandler);
    return () => window.removeEventListener("payrolls-updated", refreshHandler);
  }, []);

  // === Διαγραφή περιόδου ===
  const handleDeleteConfirm = async () => {
    if (!payrollToDelete) return;
    try {
      await api.delete(`/payrolls/${payrollToDelete}`);
      toast({ title: "Η περίοδος διαγράφηκε επιτυχώς", status: "success" });
      loadPayrolls();
    } catch (err) {
      console.error("❌ Error deleting payroll:", err);
      toast({ title: "Σφάλμα διαγραφής περιόδου", status: "error" });
    } finally {
      setConfirmOpen(false);
      setPayrollToDelete(null);
    }
  };

  if (loading)
    return (
      <Box textAlign="center" py={20}>
        <Spinner size="xl" />
        <Text mt={2}>Φόρτωση δεδομένων...</Text>
      </Box>
    );

  return (
    <VStack align="stretch" spacing={6} bg="gray.50" p={6} minH="100vh">
      {/* === Επικεφαλίδα === */}
      <HStack justify="space-between">
        <HStack>
          <Button colorScheme="red" onClick={() => navigate(-1)}>
            ⬅️ Πίσω
          </Button>
          <Heading size="lg">💶 Μισθοδοσία {searchYear}</Heading>
        </HStack>

        <Button colorScheme="teal" onClick={() => navigate("/payrolls/new")}>
          ➕ Νέα Μισθοδοσία
        </Button>
      </HStack>

      {/* === Φίλτρα Αναζήτησης === */}
      <HStack spacing={3}>
        <Input
          placeholder="Αναζήτηση υπαλλήλου..."
          value={searchName}
          onChange={(e) => setSearchName(e.target.value)}
          maxW="250px"
        />

        <Select
          placeholder="Επιλογή περιόδου"
          value={searchPeriod}
          onChange={(e) => setSearchPeriod(e.target.value)}
          maxW="200px"
        >
          {greekPeriods.map((period) => (
            <option key={period} value={period}>
              {period}
            </option>
          ))}
        </Select>

        <Input
          type="number"
          placeholder="Έτος"
          value={searchYear}
          onChange={(e) => setSearchYear(e.target.value)}
          maxW="120px"
        />

        <Button colorScheme="blue" onClick={loadPayrolls}>
          🔍 Αναζήτηση
        </Button>

        <Button
          colorScheme="gray"
          variant="outline"
          onClick={() => {
            setSearchName("");
            setSearchPeriod("");
            setSearchYear(new Date().getFullYear().toString());
            loadPayrolls();
          }}
        >
          ♻️ Επαναφορά
        </Button>
      </HStack>

      {/* === Πίνακας Μισθοδοσιών === */}
      {payrolls.length === 0 ? (
        <Text textAlign="center" color="gray.500" py={10}>
          Δεν υπάρχουν εγγραφές για τα κριτήρια που δόθηκαν.
        </Text>
      ) : (
        <Box overflowX="auto" bg="white" p={4} rounded="md" shadow="sm">
          <Table variant="simple" size="sm">
            <Thead bg="gray.100">
              <Tr>
                <Th>Υπάλληλος</Th>
                <Th>Περίοδος</Th>
                <Th isNumeric>Πληρωτέο (€)</Th>
                <Th isNumeric>Πληρωμένο (€)</Th>
                <Th isNumeric>Υπόλοιπο (€)</Th>
                <Th>Κατάσταση</Th>
                <Th textAlign="center">Ενέργειες</Th>
              </Tr>
            </Thead>

            <Tbody>
              {payrolls.map((p) => {
                const fullName =
                  p.employee_name ||
                  `${p.last_name || ""} ${p.first_name || ""}`.trim() ||
                  "-";

                const totalDue = Number(p.total_due || 0);
                const totalPaid = Number(p.total_paid || 0);
                const remaining = Number(
                  p.remaining ?? totalDue - totalPaid
                );

                const settled = remaining <= 0;
                const partial = !settled && totalPaid > 0;

                return (
                  <Tr key={p.id}>
                    <Td fontWeight="medium">{fullName}</Td>
                    <Td>{p.period}</Td>
                    <Td isNumeric>{totalDue.toFixed(2)}</Td>
                    <Td isNumeric>{totalPaid.toFixed(2)}</Td>
                    <Td isNumeric>
                      <Text color={remaining > 0 ? "red.500" : "green.600"}>
                        {remaining.toFixed(2)}
                      </Text>
                    </Td>
                    <Td>
                      <Badge
                        colorScheme={
                          settled ? "green" : partial ? "orange" : "red"
                        }
                      >
                        {settled
                          ? "Εξοφλημένο"
                          : partial
                          ? "Μερικώς"
                          : "Απλήρωτο"}
                      </Badge>
                    </Td>
                    <Td textAlign="center">
                      <HStack justify="center" spacing={2}>
                        <Button
                          size="xs"
                          colorScheme="blue"
                          onClick={() => setSelectedPayroll(p)}
                        >
                          📊 Ανάλυση
                        </Button>
                        <Button
                          size="xs"
                          colorScheme="gray"
                          onClick={() => navigate(`/payrolls/edit/${p.id}`)}
                        >
                          ✏️ Επεξεργασία
                        </Button>
                        <Button
                          size="xs"
                          colorScheme="red"
                          onClick={() => {
                            setPayrollToDelete(p.id);
                            setConfirmOpen(true);
                          }}
                        >
                          🗑️ Διαγραφή
                        </Button>
                      </HStack>
                    </Td>
                  </Tr>
                );
              })}
            </Tbody>
          </Table>
        </Box>
      )}

      {/* === Confirm Dialog === */}
      <ConfirmDialog
        isOpen={confirmOpen}
        onClose={() => setConfirmOpen(false)}
        onConfirm={handleDeleteConfirm}
        title="Διαγραφή Περιόδου"
        message="Είσαι σίγουρος ότι θέλεις να διαγράψεις αυτή την περίοδο μισθοδοσίας; Η ενέργεια δεν μπορεί να αναιρεθεί."
      />

      {/* === Modal Ανάλυσης === */}
      {selectedPayroll && (
        <PayrollItemsModal
          payroll={selectedPayroll}
          onClose={() => setSelectedPayroll(null)}
        />
      )}
    </VStack>
  );
}
