import {
  Box,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Button,
  HStack,
  VStack,
  Heading,
  Spinner,
  Text,
  useToast,
  Tag,
} from "@chakra-ui/react";
import { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import api from "../utils/axios";

export default function PaymentsDashboard() {
  const [payments, setPayments] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const toast = useToast();
  const navigate = useNavigate();

  const fetchPayments = async () => {
    try {
      setLoading(true);
      const res = await api.get("/invoice-payments"); // ✅ Παίρνει όλες τις πληρωμές πελατών
      setPayments(res.data);
    } catch (err) {
      console.error(err);
      toast({ title: "❌ Σφάλμα φόρτωσης εισπράξεων", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchPayments();
  }, []);

  const methodLabel = (method: string) => {
    switch (method) {
      case "cash":
        return "Μετρητά";
      case "bank":
        return "Τράπεζα";
      case "card":
        return "Κάρτα";
      case "elta_deposit":
        return "Κατάθεση ΕΛΤΑ";
      case "elta_withholding":
        return "Παρακράτηση ΕΛΤΑ";
      case "cod_withholding":
        return "Παρακράτηση Αντικαταβολών";
      default:
        return "Άλλο";
    }
  };

  if (loading)
    return (
      <Box textAlign="center" py={20}>
        <Spinner size="xl" />
        <Text mt={3}>Φόρτωση εισπράξεων...</Text>
      </Box>
    );

  return (
    <VStack align="stretch" spacing={6} p={6} bg="gray.50" minH="100vh">
      {/* === Header === */}
      <HStack justify="space-between" align="center">
        <HStack>
          <Button colorScheme="red" variant="outline" onClick={() => navigate(-1)}>
            ⬅ Πίσω
          </Button>
          <Heading size="lg" color="teal.600">
            💰 Εισπράξεις Πελατών
          </Heading>
        </HStack>
        <Button colorScheme="pink" onClick={() => navigate("/invoice-payments/new")}>
          ➕ Νέα Είσπραξη
        </Button>
      </HStack>

      {/* === Πίνακας Εισπράξεων === */}
      <Box bg="white" p={4} rounded="md" shadow="sm" overflowX="auto">
        <Heading size="sm" mb={3}>
          Σύνολο Εισπράξεων:{" "}
          {payments.reduce((a, b) => a + Number(b.amount || 0), 0).toFixed(2)} €
        </Heading>

        <Table variant="simple" size="sm">
          <Thead bg="gray.100">
            <Tr>
              <Th>Ημερομηνία</Th>
              <Th>Πελάτης</Th>
              <Th isNumeric>Ποσό (€)</Th>
              <Th>Τύπος Πληρωμής</Th>
              <Th>Σημειώσεις</Th>
              <Th textAlign="center">Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {payments.length > 0 ? (
              payments.map((p) => (
                <Tr key={p.id}>
                  <Td>{new Date(p.payment_date).toLocaleDateString("el-GR")}</Td>
                  <Td>{p.customer_name || "-"}</Td>

                  <Td isNumeric>{Number(p.amount).toFixed(2)}</Td>

                  <Td>
                    <Tag
                      colorScheme={
                        p.method === "cod_withholding"
                          ? "purple"
                          : p.method === "cash"
                          ? "green"
                          : p.method === "bank"
                          ? "blue"
                          : "gray"
                      }
                      variant="subtle"
                      borderRadius="full"
                    >
                      {methodLabel(p.method)}
                    </Tag>
                  </Td>

                  <Td>{p.notes || "-"}</Td>

                  <Td textAlign="center">
                    <Button
                      size="xs"
                      colorScheme="red"
                      onClick={async () => {
                        if (!window.confirm("Να διαγραφεί η είσπραξη;")) return;
                        try {
                          await api.delete(`/invoice-payments/${p.id}`);
                          toast({
                            title: "🗑️ Η είσπραξη διαγράφηκε",
                            status: "info",
                          });
                          fetchPayments();
                        } catch {
                          toast({
                            title: "❌ Σφάλμα διαγραφής",
                            status: "error",
                          });
                        }
                      }}
                    >
                      Διαγραφή
                    </Button>
                  </Td>
                </Tr>
              ))
            ) : (
              <Tr>
                <Td colSpan={6} textAlign="center">
                  Δεν υπάρχουν εισπράξεις.
                </Td>
              </Tr>
            )}
          </Tbody>
        </Table>
      </Box>
    </VStack>
  );
}
