import {
  Box,
  Button,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Input,
  VStack,
  HStack,
  Heading,
  Text,
  useToast,
  Spinner,
  IconButton,
  FormControl,
  FormLabel,
  Select,
} from "@chakra-ui/react";
import { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import api from "../utils/axios";
import { ArrowBackIcon, DeleteIcon } from "@chakra-ui/icons";

export default function InvoicePayments() {
  const navigate = useNavigate();
  const toast = useToast();

  const [payments, setPayments] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  // === Αναζήτηση πελάτη ===
  const [search, setSearch] = useState("");
  const [searchResults, setSearchResults] = useState<any[]>([]);
  const [selectedCustomer, setSelectedCustomer] = useState<any | null>(null);

  const [form, setForm] = useState({
    payment_date: "",
    amount: "",
    method: "",
    notes: "",
  });

  // === Φόρτωση πληρωμών ===
  const fetchPayments = async () => {
    try {
      setLoading(true);
      const res = await api.get(`/invoice-payments`);
      setPayments(res.data);
    } catch (err) {
      console.error(err);
      toast({ title: "❌ Σφάλμα φόρτωσης εισπράξεων", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchPayments();
  }, []);

  // === Αναζήτηση Πελάτη ===
  const handleCustomerSearch = async (term: string) => {
    setSearch(term);
    if (term.length < 2) {
      setSearchResults([]);
      return;
    }
    try {
      const res = await api.get(`/customers?search=${term}`);
      setSearchResults(res.data.data || []);
    } catch {
      setSearchResults([]);
    }
  };

  const selectCustomer = (customer: any) => {
    setSelectedCustomer(customer.id);
    setSearch(`${customer.name} (${customer.code})`);
    setSearchResults([]);
  };

  // === Ενημέρωση πεδίων ===
  const handleChange = (e: React.ChangeEvent<HTMLInputElement | HTMLSelectElement>) => {
    const { name, value } = e.target;
    setForm((prev) => ({ ...prev, [name]: value }));
  };

  // === Καταχώρηση νέας πληρωμής ===
  const handleAddPayment = async () => {
    const amount = parseFloat(form.amount.replace(",", "."));
    if (!selectedCustomer || !form.payment_date || isNaN(amount) || amount <= 0) {
      toast({
        title: "Συμπλήρωσε σωστά πελάτη, ημερομηνία και ποσό.",
        status: "warning",
      });
      return;
    }

    try {
      await api.post("/invoice-payments", {
        customer_id: selectedCustomer,
        payment_date: form.payment_date,
        amount,
        method: form.method,
        notes: form.notes,
      });
      toast({ title: "✅ Είσπραξη καταχωρήθηκε επιτυχώς", status: "success" });
      setForm({ payment_date: "", amount: "", method: "", notes: "" });
      fetchPayments();
    } catch (err) {
      console.error(err);
      toast({ title: "❌ Σφάλμα προσθήκης είσπραξης", status: "error" });
    }
  };

  // === Διαγραφή είσπραξης ===
  const handleDelete = async (id: number) => {
    if (!window.confirm("Να διαγραφεί η είσπραξη;")) return;
    try {
      await api.delete(`/invoice-payments/${id}`);
      toast({ title: "🗑️ Η είσπραξη διαγράφηκε", status: "info" });
      fetchPayments();
    } catch (err) {
      console.error(err);
      toast({ title: "❌ Σφάλμα διαγραφής", status: "error" });
    }
  };

  if (loading)
    return (
      <Box textAlign="center" py={20}>
        <Spinner size="xl" />
        <Text mt={3}>Φόρτωση εισπράξεων...</Text>
      </Box>
    );

  // === Format ελληνικής ημερομηνίας ===
  const formatDate = (dateString: string) => {
    try {
      const date = new Date(dateString);
      return date.toLocaleDateString("el-GR", {
        day: "2-digit",
        month: "2-digit",
        year: "numeric",
      });
    } catch {
      return "-";
    }
  };

  return (
    <Box maxW="6xl" mx="auto" p={6}>
      {/* Header */}
      <HStack justify="space-between" mb={6}>
        <HStack>
          <IconButton
            aria-label="Πίσω"
            icon={<ArrowBackIcon />}
            colorScheme="red"
            onClick={() => navigate(-1)}
          />
          <Heading size="lg" color="teal.600">
            💰 Εισπράξεις Πελατών
          </Heading>
        </HStack>
      </HStack>

      {/* Φόρμα νέας είσπραξης */}
      <VStack
        spacing={4}
        align="stretch"
        bg="gray.50"
        p={4}
        rounded="md"
        shadow="sm"
        mb={6}
      >
        <Heading size="md">➕ Καταχώρηση νέας είσπραξης</Heading>

        {/* Αναζήτηση πελάτη */}
        <FormControl isRequired position="relative">
          <FormLabel>Αναζήτηση Πελάτη</FormLabel>
          <Input
            placeholder="ΑΦΜ / Κωδικός / Επωνυμία"
            value={search}
            onChange={(e) => handleCustomerSearch(e.target.value)}
          />
          {searchResults.length > 0 && (
            <Box
              borderWidth="1px"
              borderRadius="md"
              mt={1}
              bg="white"
              maxH="200px"
              overflowY="auto"
              shadow="md"
              zIndex={10}
              position="absolute"
              w="100%"
            >
              {searchResults.map((c) => (
                <Box
                  key={c.id}
                  px={3}
                  py={2}
                  _hover={{ bg: "gray.100", cursor: "pointer" }}
                  onClick={() => selectCustomer(c)}
                >
                  {c.name} ({c.code}) — ΑΦΜ: {c.vat_number}
                </Box>
              ))}
            </Box>
          )}
        </FormControl>

        <HStack flexWrap="wrap" spacing={3}>
          <Input
            type="date"
            name="payment_date"
            value={form.payment_date}
            onChange={handleChange}
            lang="el"
            title="Ημερομηνία Πληρωμής"
            maxW="180px"
          />

          <Input
            type="text"
            name="amount"
            inputMode="decimal"
            value={form.amount}
            onChange={(e) => {
              const val = e.target.value.replace(/,/g, ".");
              setForm((prev) => ({ ...prev, amount: val }));
            }}
            placeholder="Ποσό (€)"
            maxW="150px"
          />

          <Select
            name="method"
            value={form.method}
            onChange={handleChange}
            placeholder="Επιλογή τρόπου πληρωμής"
            maxW="250px"
          >
            <option value="cash">Μετρητά</option>
            <option value="bank">Τράπεζα</option>
            <option value="card">Κάρτα</option>
            <option value="elta_deposit">Κατάθεση ΕΛΤΑ</option>
            <option value="elta_withholding">Παρακράτηση ΕΛΤΑ</option>
            <option value="cod_withholding">Παρακράτηση Αντικαταβολών</option>
          </Select>

          <Input
            name="notes"
            value={form.notes}
            onChange={handleChange}
            placeholder="Σημειώσεις"
            onKeyDown={(e) => e.key === "Enter" && handleAddPayment()}
            flex={1}
          />

          <Button colorScheme="green" onClick={handleAddPayment}>
            💾 Καταχώρηση
          </Button>
        </HStack>
      </VStack>

      {/* Πίνακας εισπράξεων */}
      <Box bg="white" p={4} rounded="md" shadow="sm">
        <Heading size="sm" mb={3}>
          Σύνολο Εισπράξεων:{" "}
          {payments.reduce((a, b) => a + Number(b.amount || 0), 0).toFixed(2)} €
        </Heading>

        <Table variant="striped" colorScheme="gray" size="sm">
          <Thead bg="gray.100">
            <Tr>
              <Th>Ημερομηνία</Th>
              <Th>Πελάτης</Th>
              <Th>Ποσό (€)</Th>
              <Th>Τύπος</Th>
              <Th>Σημειώσεις</Th>
              <Th>Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {payments.length > 0 ? (
              payments.map((p) => (
                <Tr key={p.id}>
                  <Td>{formatDate(p.payment_date)}</Td>
                  <Td>{p.customer_name || "-"}</Td>
                  <Td>{Number(p.amount).toFixed(2)}</Td>
                  <Td>{p.method || "-"}</Td>
                  <Td>{p.notes || "-"}</Td>
                  <Td>
                    <IconButton
                      size="sm"
                      colorScheme="red"
                      icon={<DeleteIcon />}
                      aria-label="Διαγραφή"
                      onClick={() => handleDelete(p.id)}
                    />
                  </Td>
                </Tr>
              ))
            ) : (
              <Tr>
                <Td colSpan={6} textAlign="center">
                  Δεν υπάρχουν εισπράξεις.
                </Td>
              </Tr>
            )}
          </Tbody>
        </Table>
      </Box>
    </Box>
  );
}
