import { useState, useEffect } from "react";
import {
  Box,
  Button,
  FormControl,
  FormLabel,
  Input,
  Select,
  Textarea,
  VStack,
  Heading,
  useToast,
  SimpleGrid,
  HStack,
  Tag,
  TagLabel,
} from "@chakra-ui/react";
import { useNavigate, useParams } from "react-router-dom";
import api from "../utils/axios";

export default function InvoiceForm() {
  const [invoice, setInvoice] = useState({
    customer_id: "",
    invoice_number: "",
    type: "service_invoice",
    elta_code: "",
    issue_date: new Date().toISOString().split("T")[0],
    amount: "",
    notes: "",
  });

  const [search, setSearch] = useState("");
  const [searchResults, setSearchResults] = useState<any[]>([]);
  const { id } = useParams();
  const navigate = useNavigate();
  const toast = useToast();

  // === Αν υπάρχει ID, φόρτωσε τιμολόγιο (edit mode) ===
  useEffect(() => {
    if (id) {
      api
        .get(`/invoices/${id}`)
        .then((res) => {
          const data = res.data.invoice;
          setInvoice({
            ...data,
            issue_date: data.issue_date
              ? data.issue_date.split("T")[0]
              : new Date().toISOString().split("T")[0],
          });

          // ✅ Εμφάνιση ονόματος πελάτη στο search input
          if (data.customer_name && data.customer_id) {
            setSearch(`${data.customer_name} (${data.customer_code || data.customer_id})`);
          }
        })
        .catch(() =>
          toast({
            title: "Σφάλμα",
            description: "Αποτυχία φόρτωσης τιμολογίου.",
            status: "error",
          })
        );
    }
  }, [id]);

  // === Αναζήτηση πελατών ===
  const handleCustomerSearch = async (term: string) => {
    setSearch(term);
    if (term.length < 2) {
      setSearchResults([]);
      return;
    }
    try {
      const res = await api.get(`/customers?search=${term}`);
      setSearchResults(res.data.data || []);
    } catch {
      setSearchResults([]);
    }
  };

  const selectCustomer = (customer: any) => {
    setInvoice((prev) => ({ ...prev, customer_id: customer.id }));
    setSearch(`${customer.name} (${customer.code})`);
    setSearchResults([]);
  };

  // === Καταχώρηση / Ενημέρωση ===
  const handleSubmit = async () => {
    const amount = parseFloat(
      String(invoice.amount).replace(",", ".")
    );

    if (
      !invoice.customer_id ||
      !invoice.invoice_number ||
      isNaN(amount) ||
      amount <= 0
    ) {
      toast({
        title: "Συμπλήρωσε όλα τα απαραίτητα πεδία.",
        status: "warning",
      });
      return;
    }

    const payload = { ...invoice, amount };

    try {
      if (id) {
        await api.put(`/invoices/${id}`, payload);
        toast({
          title: "✅ Ενημερώθηκε",
          description: "Το τιμολόγιο ενημερώθηκε επιτυχώς.",
          status: "success",
        });
      } else {
        await api.post("/invoices", payload);
        toast({
          title: "✅ Καταχωρήθηκε",
          description: "Το τιμολόγιο δημιουργήθηκε επιτυχώς.",
          status: "success",
        });
      }

      navigate("/invoices");
    } catch (err: any) {
      const msg =
        err.response?.data?.message ||
        "Πρόβλημα κατά την αποθήκευση τιμολογίου.";
      toast({
        title: "❌ Σφάλμα αποθήκευσης",
        description: msg,
        status: "error",
      });
    }
  };

  // === Γενική ενημέρωση πεδίων ===
  const handleChange = (e: any) => {
    const { name, value } = e.target;
    setInvoice((prev) => ({
      ...prev,
      [name]: value,
    }));
  };

  return (
    <Box maxW="6xl" mx="auto" p={6} bg="gray.50" minH="100vh" position="relative">
      <HStack justify="space-between" align="center" mb={4}>
        <Heading size="lg" color="teal.600">
          {id ? "✏️ Επεξεργασία Τιμολογίου" : "➕ Νέο Τιμολόγιο"}
        </Heading>
        {invoice.type && (
          <Tag
            size="lg"
            colorScheme={invoice.type === "credit_invoice" ? "pink" : "blue"}
            borderRadius="full"
          >
            <TagLabel>
              {invoice.type === "credit_invoice"
                ? "Πιστωτικό Τιμολόγιο"
                : "ΤΠΥ - Τιμολόγιο Παροχής Υπηρεσιών"}
            </TagLabel>
          </Tag>
        )}
      </HStack>

      <VStack align="stretch" spacing={6}>
        <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4}>
          {/* === Πελάτης === */}
          <FormControl isRequired position="relative">
            <FormLabel>Αναζήτηση Πελάτη</FormLabel>
            <Input
              placeholder="ΑΦΜ / Κωδικός / Επωνυμία"
              value={search}
              onChange={(e) => handleCustomerSearch(e.target.value)}
            />
            {searchResults.length > 0 && (
              <Box
                borderWidth="1px"
                borderRadius="md"
                mt={1}
                bg="white"
                maxH="200px"
                overflowY="auto"
                shadow="md"
                zIndex={10}
                position="absolute"
                w="100%"
              >
                {searchResults.map((c) => (
                  <Box
                    key={c.id}
                    px={3}
                    py={2}
                    _hover={{ bg: "gray.100", cursor: "pointer" }}
                    onClick={() => selectCustomer(c)}
                  >
                    {c.name} ({c.code}) — ΑΦΜ: {c.vat_number}
                  </Box>
                ))}
              </Box>
            )}
          </FormControl>

          {/* === Αριθμός Τιμολογίου === */}
          <FormControl isRequired>
            <FormLabel>Αριθμός Τιμολογίου</FormLabel>
            <Input
              name="invoice_number"
              value={invoice.invoice_number}
              onChange={handleChange}
            />
          </FormControl>

          {/* === Τύπος === */}
          <FormControl>
            <FormLabel>Τύπος Τιμολογίου</FormLabel>
            <Select name="type" value={invoice.type} onChange={handleChange}>
              <option value="service_invoice">
                ΤΠΥ (Τιμολόγιο Παροχής Υπηρεσιών)
              </option>
              <option value="credit_invoice">Πιστωτικό Τιμολόγιο</option>
            </Select>
          </FormControl>

          {/* === Ημερομηνία === */}
          <FormControl>
            <FormLabel>Ημερομηνία Έκδοσης</FormLabel>
            <Input
              type="date"
              name="issue_date"
              value={invoice.issue_date}
              onChange={handleChange}
            />
          </FormControl>

          {/* === Ποσό === */}
          <FormControl>
            <FormLabel>Ποσό (€)</FormLabel>
            <Input
              type="text"
              name="amount"
              value={invoice.amount}
              onChange={(e) => {
                const val = e.target.value.replace(/,/g, ".");
                handleChange({ target: { name: "amount", value: val } });
              }}
              placeholder="π.χ. 123,45 ή 123.45"
            />
          </FormControl>
        </SimpleGrid>

        {/* === Παρατηρήσεις === */}
        <FormControl>
          <FormLabel>Παρατηρήσεις</FormLabel>
          <Textarea
            name="notes"
            value={invoice.notes || ""}
            onChange={handleChange}
            rows={4}
          />
        </FormControl>

        {/* === Κουμπιά === */}
        <HStack justify="center" spacing={4}>
          <Button colorScheme="green" onClick={handleSubmit}>
            💾 Αποθήκευση
          </Button>
          <Button colorScheme="red" variant="outline" onClick={() => navigate(-1)}>
            ⬅ Πίσω
          </Button>
        </HStack>
      </VStack>
    </Box>
  );
}
