import {
  Box,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Button,
  HStack,
  VStack,
  Heading,
  Spinner,
  Text,
  useToast,
  Tag,
} from "@chakra-ui/react";
import { useEffect, useState } from "react";
import { useNavigate } from "react-router-dom";
import api from "../utils/axios";

export default function InvoiceDashboard() {
  const [invoices, setInvoices] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const toast = useToast();
  const navigate = useNavigate();

  const fetchInvoices = async () => {
    try {
      setLoading(true);
      const res = await api.get("/invoices");
      setInvoices(res.data);
    } catch {
      toast({ title: "❌ Σφάλμα φόρτωσης τιμολογίων", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchInvoices();
  }, []);

  if (loading)
    return (
      <Box textAlign="center" py={20}>
        <Spinner size="xl" />
        <Text mt={3}>Φόρτωση τιμολογίων...</Text>
      </Box>
    );

  return (
    <VStack align="stretch" spacing={6} p={6} bg="gray.50" minH="100vh">
      {/* === Header === */}
      <HStack justify="space-between" align="center">
        <HStack>
          <Button
            colorScheme="red"
            variant="outline"
            onClick={() => navigate(-1)}
          >
            ⬅ Πίσω
          </Button>
          <Heading size="lg" color="teal.600">
            📄 Τιμολόγια Πελατών
          </Heading>
        </HStack>
        <Button colorScheme="teal" onClick={() => navigate("/invoices/new")}>
          ➕ Νέο Τιμολόγιο
        </Button>
      </HStack>

      {/* === Πίνακας Τιμολογίων === */}
      <Box bg="white" p={4} rounded="md" shadow="sm" overflowX="auto">
        <Table variant="simple" size="sm">
          <Thead bg="gray.100">
            <Tr>
              <Th>Αριθμός</Th>
              <Th>Ημ/νία</Th>
              <Th>Πελάτης</Th>
              <Th>Τύπος</Th>
              <Th isNumeric>Ποσό (€)</Th>
              <Th>Κατάσταση</Th>
              <Th textAlign="center">Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {invoices.length > 0 ? (
              invoices.map((inv) => (
                <Tr key={inv.id}>
                  <Td>{inv.invoice_number}</Td>
                  <Td>{new Date(inv.issue_date).toLocaleDateString("el-GR")}</Td>
                  <Td>{inv.customer_name}</Td>

                  {/* === Τύπος Τιμολογίου === */}
                  <Td>
                    <Tag
                      colorScheme={
                        inv.type === "credit_invoice" ? "pink" : "blue"
                      }
                      variant="subtle"
                      borderRadius="full"
                    >
                      {inv.type === "credit_invoice"
                        ? "Πιστωτικό"
                        : "ΤΠΥ (Υπηρεσιών)"}
                    </Tag>
                  </Td>

                  <Td isNumeric>{Number(inv.amount).toFixed(2)}</Td>
                  <Td>
                    <Tag
                      colorScheme={
                        inv.status === "Εξοφλημένο"
                          ? "green"
                          : inv.status === "Μερικώς"
                          ? "yellow"
                          : "red"
                      }
                    >
                      {inv.status}
                    </Tag>
                  </Td>
                  <Td>
                    <HStack justify="center">
                      <Button
                        size="xs"
                        colorScheme="blue"
                        onClick={() => navigate(`/invoices/edit/${inv.id}`)}
                      >
                        ✏️ Επεξεργασία
                      </Button>
                      <Button
                        size="xs"
                        colorScheme="red"
                        onClick={async () => {
                          if (!window.confirm("Να διαγραφεί το τιμολόγιο;")) return;
                          try {
                            await api.delete(`/invoices/${inv.id}`);
                            toast({
                              title: "🗑️ Το τιμολόγιο διαγράφηκε",
                              status: "info",
                            });
                            fetchInvoices();
                          } catch {
                            toast({
                              title: "❌ Σφάλμα διαγραφής",
                              status: "error",
                            });
                          }
                        }}
                      >
                        Διαγραφή
                      </Button>
                    </HStack>
                  </Td>
                </Tr>
              ))
            ) : (
              <Tr>
                <Td colSpan={7} textAlign="center">
                  Δεν υπάρχουν τιμολόγια.
                </Td>
              </Tr>
            )}
          </Tbody>
        </Table>
      </Box>
    </VStack>
  );
}
