import { useEffect, useState } from "react";
import {
  Box,
  Heading,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Spinner,
  Button,
  IconButton,
  useToast,
  Badge,
  HStack,
  Text,
  VStack,
} from "@chakra-ui/react";
import { AddIcon, DeleteIcon, EditIcon } from "@chakra-ui/icons";
import api from "../utils/axios";
import { useNavigate } from "react-router-dom";

export default function EmployeesPage() {
  const [employees, setEmployees] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const toast = useToast();
  const navigate = useNavigate();

  const fetchData = async () => {
    try {
      const res = await api.get("/employees");
      setEmployees(res.data);
    } catch (err) {
      console.error("Error fetching employees:", err);
      toast({ title: "Σφάλμα φόρτωσης δεδομένων", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
  }, []);

  if (loading)
    return (
      <Box textAlign="center" py={10}>
        <Spinner size="lg" />
      </Box>
    );

  const formatEmploymentType = (type: string) => {
    switch (type) {
      case "full_time":
        return "Πλήρης";
      case "part_time":
        return "Μερική";
      case "contract":
        return "Σύμβαση";
      case "intern":
        return "Πρακτική";
      default:
        return "-";
    }
  };

  const formatEmploymentStatus = (status: string) => {
    const colors: any = {
      active: "green",
      on_leave: "orange",
      inactive: "gray",
      terminated: "red",
    };
    const labels: any = {
      active: "Ενεργός",
      on_leave: "Σε Άδεια",
      inactive: "Ανενεργός",
      terminated: "Αποχωρήσας",
    };
    return <Badge colorScheme={colors[status] || "gray"}>{labels[status] || "-"}</Badge>;
  };

  return (
    <Box minH="100vh" bg="gray.50" p={6}>
      <VStack align="stretch" spacing={6}>
        {/* Header */}
        <HStack justify="space-between">
          <Heading size="lg">👥 Εργαζόμενοι</Heading>
          <Button
            leftIcon={<AddIcon />}
            colorScheme="blue"
            onClick={() => navigate("/employees/new")}
          >
            Προσθήκη Εργαζομένου
          </Button>
        </HStack>

        {/* Table */}
        {employees.length === 0 ? (
          <Text color="gray.500" textAlign="center" mt={10}>
            Δεν υπάρχουν καταχωρημένοι υπάλληλοι.
          </Text>
        ) : (
          <Table variant="striped" colorScheme="gray" size="sm">
            <Thead>
              <Tr>
                <Th>Ονοματεπώνυμο</Th>
                <Th>Username</Th>
                <Th>Θέση</Th>
                <Th>Τμήμα</Th>
                <Th>Υποκατάστημα</Th>
                <Th>Ημ. Πρόσληψης</Th>
                <Th isNumeric>Μισθός (€)</Th>
                <Th>Τύπος</Th>
                <Th>Κατάσταση</Th>
                <Th textAlign="center">Ενέργειες</Th>
              </Tr>
            </Thead>
            <Tbody>
              {employees.map((e) => (
                <Tr key={e.id}>
                  <Td fontWeight="medium">
                    {`${e.last_name || ""} ${e.first_name || ""}`.trim() || "-"}
                  </Td>
                  <Td color="gray.600">{e.username || "-"}</Td>
                  <Td>{e.position || "-"}</Td>
                  <Td>{e.department || "-"}</Td>
                  <Td>{e.branch_name || "-"}</Td>
                  <Td>
                    {e.hire_date
                      ? new Date(e.hire_date).toLocaleDateString("el-GR", {
                          day: "2-digit",
                          month: "2-digit",
                          year: "numeric",
                        })
                      : "-"}
                  </Td>
                  <Td isNumeric>
                    {e.salary ? Number(e.salary).toFixed(2) + " €" : "-"}
                  </Td>
                  <Td>{formatEmploymentType(e.employment_type)}</Td>
                  <Td>{formatEmploymentStatus(e.employment_status)}</Td>
                  <Td>
                    <HStack justify="center">
                      <IconButton
                        icon={<EditIcon />}
                        size="sm"
                        colorScheme="blue"
                        onClick={() => navigate(`/employees/edit/${e.id}`)}
                        aria-label="Επεξεργασία"
                      />
                      <IconButton
                        icon={<DeleteIcon />}
                        size="sm"
                        colorScheme="red"
                        aria-label="Διαγραφή"
                        onClick={async () => {
                          if (window.confirm(`Διαγραφή ${e.first_name || e.username};`)) {
                            try {
                              await api.delete(`/employees/${e.id}`);
                              fetchData();
                              toast({
                                title: "Ο υπάλληλος διαγράφηκε",
                                status: "info",
                              });
                            } catch (err) {
                              toast({
                                title: "Σφάλμα διαγραφής",
                                status: "error",
                              });
                            }
                          }
                        }}
                      />
                    </HStack>
                  </Td>
                </Tr>
              ))}
            </Tbody>
          </Table>
        )}

        {/* Back button */}
        <Box textAlign="center" pt={6}>
          <Button
            colorScheme="red"
            onClick={() => {
              if (window.history.length > 2) navigate(-1);
              else navigate("/human-resources");
            }}
            width={{ base: "100%", sm: "200px" }}
          >
            ⬅ Πίσω
          </Button>
        </Box>
      </VStack>
    </Box>
  );
}
