import {
  Box,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Button,
  HStack,
  VStack,
  Heading,
  useToast,
  Spinner,
  Text,
  Input,
  IconButton,
} from "@chakra-ui/react";
import { useEffect, useState } from "react";
import api from "../utils/axios";
import { useNavigate } from "react-router-dom";
import { ArrowBackIcon } from "@chakra-ui/icons";

function formatDate(dateString: string | null): string {
  if (!dateString) return "-";
  const d = new Date(dateString);
  if (isNaN(d.getTime())) return "-";
  return d.toLocaleDateString("el-GR", {
    day: "2-digit",
    month: "2-digit",
    year: "numeric",
  });
}

export default function CustomerList() {
  const [customers, setCustomers] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);
  const [search, setSearch] = useState("");
  const [page, setPage] = useState(1);
  const [totalPages, setTotalPages] = useState(1);
  const toast = useToast();
  const navigate = useNavigate();

  const fetchCustomers = async () => {
    try {
      setLoading(true);
      const res = await api.get(`/customers?search=${search}&page=${page}`);
      setCustomers(res.data.data);
      setTotalPages(res.data.totalPages);
    } catch {
      toast({ title: "Σφάλμα φόρτωσης πελατών", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchCustomers();
  }, [page]);

  const handleSearch = () => {
    setPage(1);
    fetchCustomers();
  };

  const handleDelete = async (id: number) => {
    if (!window.confirm("Θέλεις σίγουρα να διαγράψεις τον πελάτη;")) return;
    try {
      await api.delete(`/customers/${id}`);
      toast({ title: "Ο πελάτης διαγράφηκε", status: "info" });
      fetchCustomers();
    } catch {
      toast({ title: "Σφάλμα διαγραφής", status: "error" });
    }
  };

  if (loading)
    return (
      <Box textAlign="center" py={20}>
        <Spinner size="xl" />
        <Text mt={3}>Φόρτωση πελατών...</Text>
      </Box>
    );

  return (
    <VStack align="stretch" spacing={6} bg="gray.50" p={6} minH="100vh" w="100%">
      {/* === Header === */}
      <HStack justify="space-between" align="center">
        <HStack spacing={3}>
          <IconButton
            aria-label="Πίσω"
            icon={<ArrowBackIcon />}
            colorScheme="red"
            onClick={() => navigate(-1)}
          />
          <Heading size="lg">📋 Κατάλογος Πελατών</Heading>
        </HStack>

        <HStack spacing={3}>
          <Button colorScheme="blue" onClick={() => navigate("/customers/import")}>
            📤 Import ΕΛΤΑ
          </Button>
          <Button colorScheme="teal" onClick={() => navigate("/customers/new")}>
            ➕ Προσθήκη Πελάτη
          </Button>
        </HStack>
      </HStack>

      {/* === Αναζήτηση === */}
      <HStack spacing={3} w="100%">
        <Input
          placeholder="Αναζήτηση με Επωνυμία, Κωδικό ή Α.Φ.Μ."
          value={search}
          onChange={(e) => setSearch(e.target.value)}
          onKeyDown={(e) => e.key === "Enter" && handleSearch()}
          bg="white"
        />
        <Button colorScheme="blue" onClick={handleSearch}>
          🔍 Αναζήτηση
        </Button>
        {search && (
          <Button
            colorScheme="gray"
            onClick={() => {
              setSearch("");
              setPage(1);
              fetchCustomers();
            }}
          >
            ❌ Καθαρισμός
          </Button>
        )}
      </HStack>

      {/* === Πίνακας Πελατών === */}
      <Box
        overflowX="auto"
        bg="white"
        p={4}
        rounded="md"
        shadow="sm"
        w="100%"
      >
        <Table variant="striped" colorScheme="gray" size="sm">
          <Thead bg="gray.100">
            <Tr>
              <Th minW="120px">Κωδικός</Th>
              <Th minW="250px">Επωνυμία</Th>
              <Th minW="120px">Α.Φ.Μ.</Th>
              <Th minW="220px">Τηλέφωνα</Th>
              <Th minW="150px">Πόλη</Th>
              <Th minW="150px">Ημ/νία Σύμβασης</Th>
              <Th minW="180px">Σταθμός Είσπραξης</Th>
              <Th minW="180px">Area Manager</Th>
              <Th minW="180px" textAlign="center">
                Ενέργειες
              </Th>
            </Tr>
          </Thead>
          <Tbody>
            {customers.map((c) => (
              <Tr key={c.id}>
                <Td fontFamily="mono">{c.code}</Td>
                <Td>{c.name}</Td>
                <Td>{c.vat_number}</Td>
                <Td>
                  {[c.phone1, c.phone2, c.phone3, c.phone4]
                    .filter(Boolean)
                    .join(" / ") || "-"}
                </Td>
                <Td>{c.city || "-"}</Td>
                <Td>{formatDate(c.contract_date)}</Td>
                <Td>{c.collection_station || "-"}</Td>
                <Td>{c.area_manager || "-"}</Td>
                <Td>
                  <HStack justify="center">
                    <Button
                      size="xs"
                      colorScheme="gray"
                      onClick={() => navigate(`/customers/edit/${c.id}`)}
                    >
                      ✏️ Επεξεργασία
                    </Button>
                    <Button
                      size="xs"
                      colorScheme="red"
                      onClick={() => handleDelete(c.id)}
                    >
                      🗑️ Διαγραφή
                    </Button>
                  </HStack>
                </Td>
              </Tr>
            ))}
          </Tbody>
        </Table>
      </Box>

      {/* === Pagination === */}
      <HStack justify="center" spacing={4} pt={4}>
        <Button
          onClick={() => setPage((p) => Math.max(p - 1, 1))}
          isDisabled={page === 1}
        >
          ⬅️ Προηγούμενη
        </Button>
        <Text>
          Σελίδα {page} από {totalPages}
        </Text>
        <Button
          onClick={() => setPage((p) => Math.min(p + 1, totalPages))}
          isDisabled={page === totalPages}
        >
          Επόμενη ➡️
        </Button>
      </HStack>
    </VStack>
  );
}
