import {
  Box,
  VStack,
  SimpleGrid,
  FormControl,
  FormLabel,
  Input,
  Button,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Text,
  useToast,
  Heading,
  Divider,
  Badge,
  HStack,
} from "@chakra-ui/react";
import { useState, useEffect } from "react";
import api from "../utils/axios";
import dayjs from "dayjs";
import "dayjs/locale/el";
import jsPDF from "jspdf";
import autoTable from "jspdf-autotable";
import * as XLSX from "xlsx";
import { saveAs } from "file-saver";
import { useNavigate } from "react-router-dom";
import { ArrowBackIcon } from "@chakra-ui/icons";
import { NotoSansBase64 } from "../utils/NotoSansBase64";


dayjs.locale("el");

export default function CustomerLedger() {
  const [query, setQuery] = useState("");
  const [suggestions, setSuggestions] = useState<any[]>([]);
  const [from, setFrom] = useState("");
  const [to, setTo] = useState("");
  const [data, setData] = useState<any | null>(null);
  const [loading, setLoading] = useState(false);
  const toast = useToast();
  const navigate = useNavigate();

  // === Live Search (debounce) ===
  useEffect(() => {
    const timer = setTimeout(async () => {
      if (!query.trim() || query.length < 2) {
        setSuggestions([]);
        return;
      }
      try {
        const res = await api.get(`/customers?search=${query}`);
        setSuggestions(res.data.data || res.data || []);
      } catch (err) {
        console.error("❌ Error fetching suggestions:", err);
        setSuggestions([]);
      }
    }, 400);
    return () => clearTimeout(timer);
  }, [query]);

  // === Ανάκτηση Καρτέλας ===
  const fetchLedger = async (selectedQuery?: string) => {
    const q = selectedQuery || query;
    if (!q.trim()) {
      toast({
        title: "Συμπληρώστε όνομα, κωδικό ή ΑΦΜ πελάτη",
        status: "warning",
      });
      return;
    }

    setLoading(true);
    try {
      const params = new URLSearchParams();
      params.append("q", q);
      if (from) params.append("from", from);
      if (to) params.append("to", to);

      const res = await api.get(`/ledger/search?${params.toString()}`);

      const cleaned = {
        ...res.data,
        opening_balance: Number(res.data.opening_balance) || 0,
        closing_balance: Number(res.data.closing_balance) || 0,
        total_debit: Number(res.data.total_debit) || 0,
        total_credit: Number(res.data.total_credit) || 0,
        entries: (res.data.entries || []).map((e: any) => ({
          ...e,
          debit: Number(e.debit) || 0,
          credit: Number(e.credit) || 0,
          balance: Number(e.balance) || 0,
        })),
      };

      setData(cleaned);
      setSuggestions([]);
    } catch (err) {
      console.error("❌ Error fetching ledger:", err);
      toast({ title: "Σφάλμα φόρτωσης καρτέλας", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  // === Εξαγωγή σε PDF ===
 const previewPDF = () => {
  if (!data)
    return toast({ title: "Δεν υπάρχουν δεδομένα", status: "warning" });

  const doc = new jsPDF({
    orientation: "landscape",
    unit: "mm",
    format: "a4",
    putOnlyUsedFonts: true,
    floatPrecision: 16,
  });

  // ✅ Φόρτωση UTF-8 γραμματοσειράς (NotoSans)
  doc.addFileToVFS("NotoSans-Regular.ttf", NotoSansBase64);
  doc.addFont("NotoSans-Regular.ttf", "NotoSans", "normal");
  doc.addFont("NotoSans-Regular.ttf", "NotoSans", "bold"); // ✅ δηλώνουμε και bold
  doc.setFont("NotoSans", "normal");
  doc.setFontSize(10);

  // === HEADER ===
  doc.setFontSize(16);
  doc.text(`Καρτέλα Πελάτη: ${data.customer.name}`, 14, 15);
  doc.setFontSize(10);
  doc.text(
    `Κωδικός: ${data.customer.code || "-"} | ΑΦΜ: ${
      data.customer.vat_number || "-"
    }`,
    14,
    22
  );
  doc.text(`Περίοδος: ${from || "-"} έως ${to || "-"}`, 280, 22, {
    align: "right",
  });

  // === TABLE DATA ===
  const tableData = data.entries.map((e: any) => [
    dayjs(e.txn_date).format("DD/MM/YYYY"),
    e.reference_number || "-",
    e.description || "-",
    e.debit > 0 ? e.debit.toFixed(2) : "",
    e.credit > 0 ? e.credit.toFixed(2) : "",
    e.balance.toFixed(2),
  ]);

  // === TABLE ===
  autoTable(doc, {
    head: [[
      "Ημ/νία",
      "Παραστατικό",
      "Περιγραφή",
      "Χρέωση (€)",
      "Πίστωση (€)",
      "Υπόλοιπο (€)",
    ]],
    body: [
      ["", "", "Υπόλοιπο Έναρξης", "", "", data.opening_balance.toFixed(2)],
      ...tableData,
      [
        "",
        "",
        "Σύνολα Περιόδου",
        data.total_debit.toFixed(2),
        data.total_credit.toFixed(2),
        "",
      ],
      ["", "", "Υπόλοιπο Λήξης", "", "", data.closing_balance.toFixed(2)],
    ],
    startY: 30,
    theme: "grid",

    // === ΣΤΥΛ ΚΑΙ ΕΝΟΠΟΙΗΣΗ FONT ===
    headStyles: {
      fillColor: [41, 128, 185],
      textColor: 255,
      font: "NotoSans",
      fontStyle: "bold",
      halign: "center",
    },
    bodyStyles: {
      font: "NotoSans",
      fontSize: 9,
    },
    styles: {
      font: "NotoSans",
      fontSize: 9,
      overflow: "linebreak",
      cellPadding: 2,
    },
    columnStyles: {
      0: { halign: "center" },
      1: { halign: "center" },
      2: { halign: "left" },
    },

    // ✅ ΕΠΙΒΑΛΛΕΙ το font σε κάθε κελί (κεφαλίδες + σώμα)
    didParseCell: (data) => {
      data.cell.styles.font = "NotoSans";
    },
  });

  // === FOOTER (Αρίθμηση σελίδων) ===
  const pageCount = (doc as any).internal.getNumberOfPages();
  for (let i = 1; i <= pageCount; i++) {
    doc.setPage(i);
    doc.setFont("NotoSans", "normal");
    doc.setFontSize(9);
    doc.text(
      `Σελίδα ${i} από ${pageCount}`,
      doc.internal.pageSize.getWidth() - 20,
      doc.internal.pageSize.getHeight() - 10
    );
  }

  // === ΠΡΟΕΠΙΣΚΟΠΗΣΗ PDF ===
  const pdfBlob = doc.output("blob");
  const url = URL.createObjectURL(pdfBlob);
  window.open(url, "_blank");
};


  // === Εξαγωγή σε Excel ===
  const exportToExcel = () => {
    if (!data)
      return toast({ title: "Δεν υπάρχουν δεδομένα", status: "warning" });

    const worksheetData = [
      [
        "Ημ/νία",
        "Παραστατικό",
        "Περιγραφή",
        "Χρέωση (€)",
        "Πίστωση (€)",
        "Υπόλοιπο (€)",
      ],
      ["", "", "Υπόλοιπο Έναρξης", "", "", data.opening_balance],
      ...data.entries.map((e: any) => [
        dayjs(e.txn_date).format("DD/MM/YYYY"),
        e.reference_number || "-",
        e.description || "-",
        e.debit || "",
        e.credit || "",
        e.balance,
      ]),
      ["", "", "Σύνολα Περιόδου", data.total_debit, data.total_credit, ""],
      ["", "", "Υπόλοιπο Λήξης", "", "", data.closing_balance],
    ];

    const worksheet = XLSX.utils.aoa_to_sheet(worksheetData);
    const workbook = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(workbook, worksheet, "Καρτέλα");

    const excelBuffer = XLSX.write(workbook, {
      bookType: "xlsx",
      type: "array",
    });
    const blob = new Blob([excelBuffer], {
      type: "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet",
    });
    saveAs(blob, `Καρτέλα_${data.customer.name}.xlsx`);
  };

  const formatDate = (date: string) => {
    return dayjs(date).format("DD/MM/YYYY");
  };

  return (
    <Box bg="gray.50" minH="100vh" p={6}>
      <VStack align="stretch" spacing={6}>
        <Heading size="lg">📘 Καρτέλα Πελάτη</Heading>

        <HStack justify="space-between" align="center">
          <Button
            colorScheme="red"
            leftIcon={<ArrowBackIcon />}
            onClick={() => navigate(-1)}
          >
            Πίσω
          </Button>

          <HStack spacing={3}>
            <Button colorScheme="purple" onClick={previewPDF}>
              👁️ Προεπισκόπηση PDF
            </Button>
            <Button colorScheme="green" onClick={exportToExcel}>
              📊 Εξαγωγή Excel
            </Button>
          </HStack>
        </HStack>

        {/* === Ενότητα Αναζήτησης === */}
        <Box bg="white" p={4} rounded="md" shadow="sm">
          <SimpleGrid columns={{ base: 1, md: 4 }} spacing={4}>
            <FormControl isRequired position="relative">
              <FormLabel fontWeight="semibold">Αναζήτηση Πελάτη</FormLabel>
              <Input
                placeholder="ΑΦΜ / Κωδικός / Επωνυμία"
                value={query}
                onChange={(e) => setQuery(e.target.value)}
                bg="white"
              />
              {suggestions.length > 0 && (
                <Box
                  borderWidth="1px"
                  borderRadius="md"
                  mt={1}
                  bg="white"
                  maxH="220px"
                  overflowY="auto"
                  shadow="lg"
                  zIndex={20}
                  position="absolute"
                  w="100%"
                >
                  {suggestions.map((c: any) => (
                    <Box
                      key={c.id}
                      px={3}
                      py={2}
                      _hover={{ bg: "gray.100", cursor: "pointer" }}
                      onClick={() => {
                        setQuery(`${c.name} (${c.code})`);
                        setSuggestions([]);
                      }}
                    >
                      <Text fontWeight="medium">{c.name}</Text>
                      <Text fontSize="sm" color="gray.500">
                        Κωδικός: {c.code} | ΑΦΜ: {c.vat_number}
                      </Text>
                    </Box>
                  ))}
                </Box>
              )}
            </FormControl>

            <FormControl>
              <FormLabel fontWeight="semibold">Από</FormLabel>
              <Input
                type="date"
                lang="el"
                value={from}
                onChange={(e) => setFrom(e.target.value)}
              />
            </FormControl>

            <FormControl>
              <FormLabel fontWeight="semibold">Έως</FormLabel>
              <Input
                type="date"
                lang="el"
                value={to}
                onChange={(e) => setTo(e.target.value)}
              />
            </FormControl>

            <Button
              colorScheme="blue"
              onClick={() => fetchLedger()}
              isLoading={loading}
              alignSelf="flex-end"
            >
              🔍 Αναζήτηση
            </Button>
          </SimpleGrid>
        </Box>

        {/* === Πίνακας Καρτέλας === */}
        {data && data.customer ? (
          <Box bg="white" p={4} rounded="md" shadow="sm">
            <VStack align="stretch" spacing={2} mb={4}>
              <Text fontSize="lg" fontWeight="bold">
                {data.customer.name}
              </Text>
              <Text fontSize="sm" color="gray.600">
                Κωδικός: {data.customer.code || "-"} | ΑΦΜ:{" "}
                {data.customer.vat_number || "-"}
              </Text>
            </VStack>

            <Divider mb={4} />

            <Box overflowX="auto">
              <Table size="sm">
                <Thead bg="gray.100">
                  <Tr>
                    <Th>Ημερομηνία</Th>
                    <Th>Παραστατικό</Th>
                    <Th>Περιγραφή</Th>
                    <Th isNumeric>Χρέωση (€)</Th>
                    <Th isNumeric>Πίστωση (€)</Th>
                    <Th isNumeric>Υπόλοιπο (€)</Th>
                  </Tr>
                </Thead>

                <Tbody>
                  <Tr bg="gray.50">
                    <Td colSpan={5} fontWeight="bold">
                      Υπόλοιπο Έναρξης
                    </Td>
                    <Td isNumeric>{data.opening_balance.toFixed(2)}</Td>
                  </Tr>

                  {data.entries.length === 0 ? (
                    <Tr>
                      <Td colSpan={6} textAlign="center" color="gray.500">
                        Δεν υπάρχουν κινήσεις για το διάστημα αυτό.
                      </Td>
                    </Tr>
                  ) : (
                    data.entries.map((e: any, i: number) => (
                      <Tr key={i}>
                        <Td>{formatDate(e.txn_date)}</Td>
                        <Td>{e.reference_number || "-"}</Td>
                        <Td>{e.description || "-"}</Td>
                        <Td isNumeric color="red.500">
                          {e.debit > 0 ? e.debit.toFixed(2) : ""}
                        </Td>
                        <Td isNumeric color="green.600">
                          {e.credit > 0 ? e.credit.toFixed(2) : ""}
                        </Td>
                        <Td isNumeric fontWeight="bold">
                          {e.balance.toFixed(2)}
                        </Td>
                      </Tr>
                    ))
                  )}

                  <Tr bg="gray.100" fontWeight="bold">
                    <Td colSpan={3}>Σύνολα Περιόδου</Td>
                    <Td isNumeric color="red.500">
                      {data.total_debit.toFixed(2)}
                    </Td>
                    <Td isNumeric color="green.600">
                      {data.total_credit.toFixed(2)}
                    </Td>
                    <Td></Td>
                  </Tr>

                  <Tr bg="gray.100" fontWeight="bold">
                    <Td colSpan={5}>Υπόλοιπο Λήξης</Td>
                    <Td isNumeric>{data.closing_balance.toFixed(2)}</Td>
                  </Tr>
                </Tbody>
              </Table>
            </Box>

            <Box mt={4}>
              <Badge
                colorScheme={
                  data.closing_balance > 0
                    ? "red"
                    : data.closing_balance < 0
                    ? "green"
                    : "gray"
                }
                fontSize="md"
                p={2}
                rounded="md"
              >
                {data.closing_balance > 0
                  ? `Υπόλοιπο Προς Είσπραξη: ${data.closing_balance.toFixed(2)} €`
                  : data.closing_balance < 0
                  ? `Πιστωτικό Υπόλοιπο: ${(data.closing_balance * -1).toFixed(
                      2
                    )} €`
                  : "Μηδενικό Υπόλοιπο"}
              </Badge>
            </Box>
          </Box>
        ) : (
          <Text color="gray.500" fontSize="sm">
            Συμπληρώστε αναζήτηση για εμφάνιση καρτέλας.
          </Text>
        )}
      </VStack>
    </Box>
  );
}
