import { useState, useEffect } from "react";
import {
  Box,
  Button,
  Input,
  FormControl,
  FormLabel,
  VStack,
  Heading,
  SimpleGrid,
  useToast,
  HStack,
} from "@chakra-ui/react";
import { useNavigate, useParams } from "react-router-dom";
import api from "../utils/axios";

export default function CustomerForm() {
  const [formData, setFormData] = useState({
    code: "",
    name: "",
    trade_name: "",
    profession: "",
    vat_number: "",
    tax_office: "",
    address: "",
    city: "",
    region: "",
    postal_code: "",
    country: "ΕΛΛΑΔΑ",
    phone1: "",
    phone2: "",
    phone3: "",
    phone4: "",
    email: "",
    iban: "",
    bank_name: "",
    contact_person: "",
    sales_station: "",
    collection_station: "",
    contract_date: "",
    credit_limit: "",
    area_manager: "",
    notes: "",
    is_active: 1,
  });

  const { id } = useParams();
  const toast = useToast();
  const navigate = useNavigate();

  useEffect(() => {
    if (id) {
      api
        .get(`/customers/${id}`)
        .then((res) => {
          const d = res.data || {};
          setFormData({
            code: d.code || "",
            name: d.name || "",
            trade_name: d.trade_name || "",
            profession: d.profession || "",
            vat_number: d.vat_number || "",
            tax_office: d.tax_office || "",
            address: d.address || "",
            city: d.city || "",
            region: d.region || "",
            postal_code: d.postal_code || "",
            country: d.country || "ΕΛΛΑΔΑ",
            phone1: d.phone1 || "",
            phone2: d.phone2 || "",
            phone3: d.phone3 || "",
            phone4: d.phone4 || "",
            email: d.email || "",
            iban: d.iban || "",
            bank_name: d.bank_name || "",
            contact_person: d.contact_person || "",
            sales_station: d.sales_station || "",
            collection_station: d.collection_station || "",
            contract_date: d.contract_date ? d.contract_date.split("T")[0] : "",
            credit_limit: d.credit_limit ?? "",
            area_manager: d.area_manager || "",
            notes: d.notes || "",
            is_active: d.is_active ?? 1,
          });
        })
        .catch(() =>
          toast({
            title: "Σφάλμα",
            description: "Αποτυχία φόρτωσης στοιχείων πελάτη.",
            status: "error",
          })
        );
    }
  }, [id]);

  const handleChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const { name, value } = e.target;
    setFormData((prev) => ({ ...prev, [name]: value }));
  };

  const handleSubmit = async () => {
    try {
      if (id) {
        await api.put(`/customers/${id}`, formData);
        toast({
          title: "✅ Ενημερώθηκε",
          description: "Τα στοιχεία του πελάτη ενημερώθηκαν.",
          status: "success",
        });
      } else {
        await api.post("/customers", formData);
        toast({
          title: "✅ Καταχωρήθηκε",
          description: "Ο πελάτης προστέθηκε επιτυχώς.",
          status: "success",
        });
      }
      navigate("/customers/list");
    } catch {
      toast({
        title: "❌ Σφάλμα",
        description: "Αποτυχία αποθήκευσης πελάτη.",
        status: "error",
      });
    }
  };

  return (
    <Box maxW="8xl" mx="auto" p={6}>
      {/* === Header με κουμπί πίσω === */}
      <HStack justify="space-between" align="center" mb={6}>
        <Heading size="lg">
          {id ? "✏️ Επεξεργασία Πελάτη" : "➕ Νέος Πελάτης"}
        </Heading>
        <Button
          colorScheme="red"
          variant="solid"
          onClick={() => navigate("/customers/list")}
        >
          ⬅️ Πίσω
        </Button>
      </HStack>

      <VStack spacing={6} align="stretch">
        {/* === Στοιχεία ταυτότητας === */}
        <SimpleGrid minChildWidth="280px" spacing={4}>
          <FormControl>
            <FormLabel>Κωδικός</FormLabel>
            <Input name="code" value={formData.code || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Επωνυμία</FormLabel>
            <Input name="name" value={formData.name || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Διακριτικός Τίτλος</FormLabel>
            <Input name="trade_name" value={formData.trade_name || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Επάγγελμα</FormLabel>
            <Input name="profession" value={formData.profession || ""} onChange={handleChange} />
          </FormControl>
        </SimpleGrid>

        {/* === Φορολογικά === */}
        <SimpleGrid minChildWidth="280px" spacing={4}>
          <FormControl>
            <FormLabel>Α.Φ.Μ.</FormLabel>
            <Input name="vat_number" value={formData.vat_number || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Δ.Ο.Υ.</FormLabel>
            <Input name="tax_office" value={formData.tax_office || ""} onChange={handleChange} />
          </FormControl>
        </SimpleGrid>

        {/* === Διεύθυνση === */}
        <SimpleGrid minChildWidth="280px" spacing={4}>
          <FormControl>
            <FormLabel>Διεύθυνση</FormLabel>
            <Input name="address" value={formData.address || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Πόλη</FormLabel>
            <Input name="city" value={formData.city || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Νομός</FormLabel>
            <Input name="region" value={formData.region || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Τ.Κ.</FormLabel>
            <Input name="postal_code" value={formData.postal_code || ""} onChange={handleChange} />
          </FormControl>
        </SimpleGrid>

        {/* === Τηλέφωνα & Email === */}
        <SimpleGrid minChildWidth="280px" spacing={4}>
          <FormControl>
            <FormLabel>Τηλέφωνο 1</FormLabel>
            <Input name="phone1" value={formData.phone1 || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Τηλέφωνο 2</FormLabel>
            <Input name="phone2" value={formData.phone2 || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Τηλέφωνο 3</FormLabel>
            <Input name="phone3" value={formData.phone3 || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Τηλέφωνο 4</FormLabel>
            <Input name="phone4" value={formData.phone4 || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Email</FormLabel>
            <Input name="email" value={formData.email || ""} onChange={handleChange} />
          </FormControl>
        </SimpleGrid>

        {/* === Οικονομικά === */}
        <SimpleGrid minChildWidth="280px" spacing={4}>
          <FormControl>
            <FormLabel>IBAN</FormLabel>
            <Input name="iban" value={formData.iban || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Τράπεζα</FormLabel>
            <Input name="bank_name" value={formData.bank_name || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Πίστωση (€)</FormLabel>
            <Input
              type="number"
              name="credit_limit"
              value={formData.credit_limit ?? ""}
              onChange={handleChange}
            />
          </FormControl>
        </SimpleGrid>

        {/* === ELTA / Σύμβαση === */}
        <SimpleGrid minChildWidth="280px" spacing={4}>
          <FormControl>
            <FormLabel>Sales Station</FormLabel>
            <Input name="sales_station" value={formData.sales_station || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Collection Station</FormLabel>
            <Input name="collection_station" value={formData.collection_station || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Ημερομηνία Σύμβασης</FormLabel>
            <Input
              type="date"
              name="contract_date"
              value={formData.contract_date || ""}
              onChange={handleChange}
            />
          </FormControl>
        </SimpleGrid>

        <SimpleGrid minChildWidth="280px" spacing={4}>
          <FormControl>
            <FormLabel>Area Manager</FormLabel>
            <Input name="area_manager" value={formData.area_manager || ""} onChange={handleChange} />
          </FormControl>
          <FormControl>
            <FormLabel>Σημειώσεις</FormLabel>
            <Input name="notes" value={formData.notes || ""} onChange={handleChange} />
          </FormControl>
        </SimpleGrid>

        <Button colorScheme="teal" width="200px" alignSelf="center" onClick={handleSubmit}>
          💾 Αποθήκευση
        </Button>
      </VStack>
    </Box>
  );
}
