import { useEffect, useState } from "react";
import { Box, Heading, Table, Thead, Tbody, Tr, Th, Td, Input, Button, Spinner, Text } from "@chakra-ui/react";
import api from "../utils/axios";
import CashExportButton from "../components/CashExportButton";

export default function CashReports() {
  const [reportData, setReportData] = useState<any>(null);
  const [date, setDate] = useState<string>(new Date().toISOString().split("T")[0]);
  const [loading, setLoading] = useState(false);

  const fetchReport = async () => {
    try {
      setLoading(true);
      const res = await api.get(`/cash/reports/daily?date=${date}`);
      setReportData(res.data);
    } catch (err) {
      console.error("Error fetching report:", err);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchReport();
  }, [date]);

  return (
    <Box p={6} bg="gray.50" minH="100vh">
      <Heading size="lg" mb={4}>Ημερήσιο Report Ταμείων</Heading>

      <Box display="flex" gap={3} mb={4}>
        <Input
          type="date"
          value={date}
          onChange={(e) => setDate(e.target.value)}
          max={new Date().toISOString().split("T")[0]}
          width="250px"
        />
        <Button colorScheme="blue" onClick={fetchReport}>
          Αναζήτηση
        </Button>
        <CashExportButton date={date} />
      </Box>

      {loading ? (
        <Spinner size="xl" />
      ) : reportData?.data?.length ? (
        <>
          <Text fontWeight="bold" mb={2}>
            Σύνολο Δηλωθέντων: {reportData.summary.total_declared.toFixed(2)} € |
            Σύνολο Συστήματος: {reportData.summary.total_system.toFixed(2)} € |
            Διαφορά: {reportData.summary.total_diff.toFixed(2)} €
          </Text>

          <Table variant="striped">
            <Thead>
              <Tr>
                <Th>Κατάστημα</Th>
                <Th>Τύπος</Th>
                <Th>Χρήστης</Th>
                <Th isNumeric>Σύστημα</Th>
                <Th isNumeric>Δηλωθέν</Th>
                <Th isNumeric>Διαφορά</Th>
              </Tr>
            </Thead>
            <Tbody>
              {reportData.data.map((row: any, idx: number) => (
                <Tr key={idx}>
                  <Td>{row.branch_name}</Td>
                  <Td>{row.cash_type}</Td>
                  <Td>{row.user_name}</Td>
                  <Td isNumeric>{row.system_cash?.toFixed(2)} €</Td>
                  <Td isNumeric>{row.declared_cash?.toFixed(2)} €</Td>
                  <Td isNumeric color={row.diff >= 0 ? "green.500" : "red.500"}>
                    {row.diff?.toFixed(2)} €
                  </Td>
                </Tr>
              ))}
            </Tbody>
          </Table>
        </>
      ) : (
        <Text color="gray.500" mt={4}>Δεν υπάρχουν δεδομένα για την ημερομηνία {date}</Text>
      )}
    </Box>
  );
}
