import {
  Box,
  Button,
  Heading,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  IconButton,
  useDisclosure,
  useToast,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalCloseButton,
  ModalBody,
  Text,
  Spinner,
  Flex,
} from "@chakra-ui/react";
import { DeleteIcon, EditIcon, ArrowBackIcon } from "@chakra-ui/icons";
import { useEffect, useState } from "react";
import axios from "../utils/axios";
import CashEntryForm from "../components/CashEntryForm";
import { useNavigate } from "react-router-dom";
import DatePicker from "react-datepicker";
import { el } from "date-fns/locale";
import "react-datepicker/dist/react-datepicker.css";

export default function CashOverview() {
  const [sessions, setSessions] = useState<any[]>([]);
  const [editing, setEditing] = useState<any | null>(null);
  const [loadingEdit, setLoadingEdit] = useState(false);
  const [loading, setLoading] = useState(false);
  const [date, setDate] = useState(new Date().toISOString().split("T")[0]);

  const { isOpen, onOpen, onClose } = useDisclosure();
  const toast = useToast();
  const navigate = useNavigate();

  const fetchSessions = async (selectedDate: string) => {
    try {
      setLoading(true);
      const res = await axios.get(`/cash-sessions?date=${selectedDate}`);
      setSessions(res.data);
    } catch (err) {
      console.error("❌ Fetch error:", err);
      toast({ title: "Σφάλμα φόρτωσης ταμείων", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchSessions(date);
  }, [date]);

  const handleDelete = async (id: number) => {
    if (!window.confirm("Είστε σίγουρος;")) return;
    try {
      await axios.delete(`/cash-sessions/${id}`);
      toast({ title: "Η εγγραφή διαγράφηκε", status: "info" });
      fetchSessions(date);
    } catch (err) {
      console.error("❌ Delete error:", err);
      toast({ title: "Σφάλμα διαγραφής", status: "error" });
    }
  };

  const calculateExpenses = (entry: any) =>
    (entry.expenses || []).reduce((sum, e) => sum + (Number(e.amount) || 0), 0);

  return (
    <Box p={4} maxW="container.lg" mx="auto" minH="100vh">
      <Flex justify="space-between" align="center" mb={4} flexWrap="wrap">
        <Heading size="md">Καταχωρήσεις Ταμείου</Heading>
        <Button
          leftIcon={<ArrowBackIcon />}
          onClick={() => navigate(-1)}
          colorScheme="gray"
        >
          Πίσω
        </Button>
      </Flex>

      {/* Επιλογή Ημερομηνίας */}
      <Flex mb={4} align="center" gap={3} wrap="wrap">
        <Text fontWeight="medium">Ημερομηνία:</Text>
        <DatePicker
          selected={new Date(date)}
          onChange={(d: Date) => {
            const iso = d.toISOString().split("T")[0];
            setDate(iso);
          }}
          locale={el}
          dateFormat="dd/MM/yyyy"
          maxDate={new Date()}
          className="chakra-input css-datepicker"
        />
        <Button colorScheme="blue" onClick={() => fetchSessions(date)}>
          Εμφάνιση
        </Button>
      </Flex>

      {loading ? (
        <Box textAlign="center" py={6}>
          <Spinner size="lg" />
          <Text mt={2}>Φόρτωση...</Text>
        </Box>
      ) : sessions.length === 0 ? (
        <Text color="gray.500" mt={4}>
          Δεν υπάρχουν καταχωρημένα ταμεία για{" "}
          {new Date(date).toLocaleDateString("el-GR")}
        </Text>
      ) : (
        <Table variant="simple" size="sm">
          <Thead>
            <Tr>
              <Th>Ημερομηνία</Th>
              <Th>Χρήστης</Th>
              <Th>Κατάστημα</Th>
              <Th>Τύπος</Th>
              <Th isNumeric>Θεωρητικό</Th>
              <Th isNumeric>Πραγματικό</Th>
              <Th isNumeric>Έξοδα</Th>
              <Th isNumeric>Διαφορά</Th>
              <Th>Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {sessions.map((s) => {
              const totalExpenses = calculateExpenses(s);
              const difference =
                Number(s.declared_cash) - Number(s.system_cash) + totalExpenses;
              return (
                <Tr key={s.id}>
                  <Td>{new Date(s.session_date).toLocaleDateString("el-GR")}</Td>
                  <Td>{s.user?.username || "-"}</Td>
                  <Td>{s.branch?.name || s.branch_id}</Td>
                  <Td>{s.cash_type}</Td>
                  <Td isNumeric>€ {Number(s.system_cash).toFixed(2)}</Td>
                  <Td isNumeric>€ {Number(s.declared_cash).toFixed(2)}</Td>
                  <Td isNumeric>€ {totalExpenses.toFixed(2)}</Td>
                  <Td
                    isNumeric
                    color={
                      difference < 0
                        ? "red.500"
                        : difference === 0
                        ? "blue.500"
                        : "green.600"
                    }
                  >
                    € {difference.toFixed(2)}
                  </Td>
                  <Td>
                    <IconButton
                      icon={<EditIcon />}
                      size="sm"
                      mr={2}
                      onClick={() => navigate(`/cash-entry/${s.id}`)}
                      aria-label="Edit"
                    />
                    <IconButton
                      icon={<DeleteIcon />}
                      size="sm"
                      colorScheme="red"
                      onClick={() => handleDelete(s.id)}
                      aria-label="Delete"
                    />
                  </Td>
                </Tr>
              );
            })}
          </Tbody>
        </Table>
      )}

      {/* Modal Επεξεργασίας */}
      <Modal isOpen={isOpen} onClose={onClose} size="xl">
        <ModalOverlay />
        <ModalContent>
          <ModalHeader>Επεξεργασία Καταχώρησης</ModalHeader>
          <ModalCloseButton />
          <ModalBody pb={6}>
            {loadingEdit ? (
              <Box textAlign="center" py={6}>
                <Spinner size="xl" />
                <Text mt={4}>Φόρτωση...</Text>
              </Box>
            ) : editing ? (
              <CashEntryForm
                initialData={editing}
                onClose={onClose}
                onSuccess={() => {
                  fetchSessions(date);
                  onClose();
                }}
              />
            ) : (
              <Text>Δεν βρέθηκε εγγραφή.</Text>
            )}
          </ModalBody>
        </ModalContent>
      </Modal>
    </Box>
  );
}
