import {
  Box,
  Button,
  Heading,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  IconButton,
  useDisclosure,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalCloseButton,
  FormControl,
  FormLabel,
  Input,
  Select,
  VStack,
  useToast,
} from "@chakra-ui/react";
import { AddIcon, DeleteIcon, EditIcon, ArrowBackIcon } from "@chakra-ui/icons";
import { useEffect, useState } from "react";
import axios from "../utils/axios";
import { useNavigate } from "react-router-dom";

interface Branch {
  id: number;
  name: string;
  address: string;
  phone: string;
  branch_type: "ELTA" | "COURIER" | "HUB";
  created_at: string;
}

export default function Branches() {
  const [branches, setBranches] = useState<Branch[]>([]);
  const [formData, setFormData] = useState<Partial<Branch>>({});
  const [editingId, setEditingId] = useState<number | null>(null);
  const { isOpen, onOpen, onClose } = useDisclosure();
  const toast = useToast();
  const navigate = useNavigate();

  const fetchBranches = async () => {
    try {
      const res = await axios.get("/branches");
      setBranches(res.data);
    } catch (err) {
      console.error("❌ Fetch error:", err);
    }
  };

  useEffect(() => {
    fetchBranches();
  }, []);

  const handleSubmit = async () => {
    try {
      if (editingId) {
        await axios.put(`/branches/${editingId}`, formData);
        toast({ title: "Ενημερώθηκε", status: "success", duration: 2000 });
      } else {
        await axios.post("/branches", formData);
        toast({ title: "Προστέθηκε", status: "success", duration: 2000 });
      }
      fetchBranches();
      onClose();
      setFormData({});
      setEditingId(null);
    } catch (err) {
      console.error("❌ Save error:", err);
      toast({ title: "Σφάλμα", status: "error", duration: 3000 });
    }
  };

  const handleEdit = (branch: Branch) => {
    setEditingId(branch.id);
    setFormData(branch);
    onOpen();
  };

  const handleDelete = async (id: number) => {
    try {
      await axios.delete(`/branches/${id}`);
      toast({ title: "Διαγράφηκε", status: "info", duration: 2000 });
      fetchBranches();
    } catch (err) {
      console.error("❌ Delete error:", err);
    }
  };

  return (
    <Box p={4} maxW="100vw" overflowX="auto">
      <Button leftIcon={<ArrowBackIcon />} onClick={() => navigate(-1)} mb={4}>
        Πίσω
      </Button>

      <Heading size={{ base: "md", md: "lg" }} mb={4} textAlign="center">
        Υποκαταστήματα
      </Heading>

      <Button
        leftIcon={<AddIcon />}
        colorScheme="green"
        onClick={() => {
          setFormData({});
          setEditingId(null);
          onOpen();
        }}
        mb={4}
      >
        Νέο Υποκατάστημα
      </Button>

      <Box overflowX="auto">
        <Table variant="simple" size="sm">
          <Thead>
            <Tr>
              <Th>Όνομα</Th>
              <Th>Διεύθυνση</Th>
              <Th>Τηλέφωνο</Th>
              <Th>Τύπος</Th>
              <Th>Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {branches.map((b) => (
              <Tr key={b.id}>
                <Td>{b.name}</Td>
                <Td>{b.address}</Td>
                <Td>{b.phone}</Td>
                <Td>{b.branch_type}</Td>
                <Td>
                  <IconButton
                    icon={<EditIcon />}
                    size="sm"
                    mr={2}
                    onClick={() => handleEdit(b)}
                    aria-label="edit"
                  />
                  <IconButton
                    icon={<DeleteIcon />}
                    size="sm"
                    colorScheme="red"
                    onClick={() => handleDelete(b.id)}
                    aria-label="delete"
                  />
                </Td>
              </Tr>
            ))}
          </Tbody>
        </Table>
      </Box>

      {/* Modal Form */}
      <Modal isOpen={isOpen} onClose={onClose} isCentered size={{ base: "full", sm: "md" }}>
        <ModalOverlay />
        <ModalContent>
          <ModalHeader textAlign="center">
            {editingId ? "Επεξεργασία" : "Νέο"} Υποκατάστημα
          </ModalHeader>
          <ModalCloseButton />
          <ModalBody>
            <VStack spacing={3} py={3}>
              <FormControl isRequired>
                <FormLabel>Όνομα</FormLabel>
                <Input
                  value={formData.name || ""}
                  onChange={(e) =>
                    setFormData((prev) => ({ ...prev, name: e.target.value }))
                  }
                  width="100%"
                />
              </FormControl>
              <FormControl>
                <FormLabel>Διεύθυνση</FormLabel>
                <Input
                  value={formData.address || ""}
                  onChange={(e) =>
                    setFormData((prev) => ({ ...prev, address: e.target.value }))
                  }
                  width="100%"
                />
              </FormControl>
              <FormControl>
                <FormLabel>Τηλέφωνο</FormLabel>
                <Input
                  value={formData.phone || ""}
                  onChange={(e) =>
                    setFormData((prev) => ({ ...prev, phone: e.target.value }))
                  }
                  width="100%"
                />
              </FormControl>
              <FormControl isRequired>
                <FormLabel>Τύπος</FormLabel>
                <Select
                  value={formData.branch_type || ""}
                  onChange={(e) =>
                    setFormData((prev) => ({
                      ...prev,
                      branch_type: e.target.value as Branch["branch_type"],
                    }))
                  }
                  width="100%"
                >
                  <option value="">-- Επιλέξτε --</option>
                  <option value="ELTA">ELTA</option>
                  <option value="COURIER">COURIER</option>
                  <option value="HUB">HUB</option>
                </Select>
              </FormControl>
              <Button onClick={handleSubmit} colorScheme="blue" width="100%">
                Αποθήκευση
              </Button>
            </VStack>
          </ModalBody>
        </ModalContent>
      </Modal>
    </Box>
  );
}
