import { useEffect, useState } from "react";
import { Box, Heading, Table, Thead, Tbody, Tr, Th, Td, Spinner, Text } from "@chakra-ui/react";
import api from "../utils/axios";

export default function VehicleStatsWidget() {
  const [data, setData] = useState<any[]>([]);
  const [loading, setLoading] = useState(true);

  const fetchData = async () => {
    try {
      const res = await api.get("/vehicles/stats/dashboard");
      setData(res.data);
      setLoading(false);
    } catch (err) {
      console.error("Error fetching vehicle stats:", err);
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData();
    const interval = setInterval(fetchData, 60000);
    return () => clearInterval(interval);
  }, []);

  if (loading)
    return (
      <Box textAlign="center" py={10}>
        <Spinner size="lg" />
      </Box>
    );

  if (!data.length)
    return (
      <Box textAlign="center" py={6}>
        <Text color="gray.400">Δεν υπάρχουν δεδομένα οχημάτων</Text>
      </Box>
    );

  const totals = data.reduce(
    (acc, row) => ({
      active: acc.active + (row.status_code === "active" ? row.total : 0),
      maintenance: acc.maintenance + (row.status_code === "maintenance" ? row.total : 0),
      inactive: acc.inactive + (row.status_code === "inactive" ? row.total : 0),
    }),
    { active: 0, maintenance: 0, inactive: 0 }
  );

  return (
    <Box bg="white" p={4} rounded="xl" shadow="md" border="1px solid" borderColor="gray.100">
      <Heading size="md" mb={4}>
        Κατάσταση Οχημάτων
      </Heading>

      <Table variant="simple" size="sm">
        <Thead>
          <Tr>
            <Th>Κατάσταση</Th>
            <Th isNumeric>Σύνολο</Th>
          </Tr>
        </Thead>
        <Tbody>
          <Tr>
            <Td color="green.600" fontWeight="bold">
              Ενεργά
            </Td>
            <Td isNumeric>{totals.active}</Td>
          </Tr>
          <Tr>
            <Td color="orange.500" fontWeight="bold">
              Σε Συντήρηση
            </Td>
            <Td isNumeric>{totals.maintenance}</Td>
          </Tr>
          <Tr>
            <Td color="red.500" fontWeight="bold">
              Ανενεργά
            </Td>
            <Td isNumeric>{totals.inactive}</Td>
          </Tr>
        </Tbody>
      </Table>
    </Box>
  );
}
