import React, { useRef, useState } from "react";
import {
  Box,
  Button,
  Input,
  Textarea,
  FormLabel,
  Switch,
  VStack,
  Text,
} from "@chakra-ui/react";
import SignatureCanvas from "react-signature-canvas";
import axios from "../utils/axios"; // Χρήση axios instance με baseURL

interface VehicleReturnFormProps {
  logbookId: number;
  onSuccess: () => void;
}

const VehicleReturnForm: React.FC<VehicleReturnFormProps> = ({
  logbookId,
  onSuccess,
}) => {
  const [returnTime, setReturnTime] = useState(
    new Date().toISOString().slice(0, 16)
  );
  const [endKm, setEndKm] = useState("");
  const [notes, setNotes] = useState("");
  const [hasIncident, setHasIncident] = useState(false);
  const sigCanvasRef = useRef<SignatureCanvas>(null);

  const handleClear = () => sigCanvasRef.current?.clear();

  const handleSubmit = async () => {
    if (sigCanvasRef.current?.isEmpty()) {
      alert("Υπογράψτε για επιστροφή.");
      return;
    }

    try {
      await axios.put(`/vehicle-logbook/${logbookId}/return`, {
        return_time: returnTime.replace("T", " "), // ✅ Χειροκίνητη ώρα
        end_km: endKm,
        notes,
        has_incident: hasIncident,
      });

      const signatureData = sigCanvasRef.current
        ?.getCanvas()
        .toDataURL("image/png");
      const blob = await (await fetch(signatureData!)).blob();
      const formData = new FormData();
      formData.append("type", "return");
      formData.append("file", blob, "return_signature.png");

      await axios.post(`/vehicle-logbook/${logbookId}/signature`, formData, {
        headers: { "Content-Type": "multipart/form-data" },
      });

      alert("Η επιστροφή καταχωρήθηκε.");
      onSuccess();
    } catch (err) {
      console.error("❌ Axios error:", err);
      alert("Σφάλμα επιστροφής.");
    }
  };

  return (
    <VStack spacing={4} align="stretch">
      <FormLabel>Ημερομηνία/Ώρα Επιστροφής</FormLabel>
      <Input
        type="datetime-local"
        value={returnTime}
        onChange={(e) => setReturnTime(e.target.value)}
      />

      <FormLabel>Χιλιόμετρα επιστροφής</FormLabel>
      <Input
        type="number"
        value={endKm}
        onChange={(e) => setEndKm(e.target.value)}
      />

      <FormLabel>Παρατηρήσεις</FormLabel>
      <Textarea value={notes} onChange={(e) => setNotes(e.target.value)} />

      <Box display="flex" alignItems="center">
        <FormLabel mb="0">Υπήρξε συμβάν;</FormLabel>
        <Switch
          isChecked={hasIncident}
          onChange={() => setHasIncident(!hasIncident)}
        />
        <Text ml={2}>{hasIncident ? "Ναι" : "Όχι"}</Text>
      </Box>

      <FormLabel>Υπογραφή Επιστροφής</FormLabel>
      <Box border="1px" borderColor="gray.300" borderRadius="md">
        <SignatureCanvas
          ref={sigCanvasRef}
          penColor="black"
          canvasProps={{ width: 300, height: 100, className: "sigCanvas" }}
        />
      </Box>
      <Button size="sm" onClick={handleClear}>
        Καθαρισμός
      </Button>

      <Button colorScheme="green" onClick={handleSubmit}>
        Καταχώρηση Επιστροφής
      </Button>
    </VStack>
  );
};

export default VehicleReturnForm;
