import React, { useRef, useState } from "react";
import SignatureCanvas from "react-signature-canvas";
import {
  Box, Button, Input, Textarea, Select, FormLabel, Switch, VStack, Text
} from "@chakra-ui/react";
import axios from "../utils/axios";
import { useAuth } from "../context/AuthContext";

interface Vehicle {
  id: number;
  plate: string;
}

interface VehiclePickupFormProps {
  vehicles: Vehicle[];
  userId: number;
  onSuccess: () => void;
}

const VehiclePickupForm: React.FC<VehiclePickupFormProps> = ({
  vehicles = [],
  userId,
  onSuccess,
}) => {
  const { user } = useAuth();
  const branchId = user?.branch_id ?? null;

  const [vehicleId, setVehicleId] = useState("");
  const [pickupTime, setPickupTime] = useState(new Date().toISOString().slice(0, 16));
  const [startKm, setStartKm] = useState("");
  const [notes, setNotes] = useState("");
  const [hasIncident, setHasIncident] = useState(false);
  const sigCanvasRef = useRef<SignatureCanvas>(null);

  const handleClear = () => sigCanvasRef.current?.clear();

  const handleSubmit = async () => {
    if (sigCanvasRef.current?.isEmpty()) {
      alert("Παρακαλώ υπογράψτε πριν υποβάλετε.");
      return;
    }

    try {
      const res = await axios.post("/vehicle-logbook", {
        vehicle_id: vehicleId,
        user_id: userId,
        branch_id: branchId,
        pickup_time: pickupTime,
        start_km: startKm,
        notes,
        has_incident: hasIncident
      });

      const logbookId = res.data.id;

      const signatureData = sigCanvasRef.current
        ?.getCanvas()
        .toDataURL("image/png");
      const blob = await (await fetch(signatureData!)).blob();
      const formData = new FormData();
      formData.append("type", "pickup");
      formData.append("file", blob, "signature.png");

      await axios.post(`/vehicle-logbook/${logbookId}/signature`, formData, {
        headers: { "Content-Type": "multipart/form-data" }
      });

      alert("Η παραλαβή καταχωρήθηκε!");
      onSuccess();
    } catch (err) {
      console.error("❌ Axios error:", err);
      alert("Σφάλμα κατά την αποθήκευση.");
    }
  };

  return (
    <VStack spacing={4} align="stretch">
      <FormLabel>Όχημα</FormLabel>
      <Select value={vehicleId} onChange={(e) => setVehicleId(e.target.value)}>
        <option value="">Επιλέξτε όχημα</option>
        {Array.isArray(vehicles) &&
          vehicles.map((v) => (
            <option key={v.id} value={v.id}>{v.plate}</option>
          ))}
      </Select>

      <FormLabel>Ημερομηνία/Ώρα</FormLabel>
      <Input
        type="datetime-local"
        value={pickupTime}
        onChange={(e) => setPickupTime(e.target.value)}
      />

      <FormLabel>Χιλιόμετρα εκκίνησης</FormLabel>
      <Input type="number" value={startKm} onChange={(e) => setStartKm(e.target.value)} />

      <FormLabel>Παρατηρήσεις</FormLabel>
      <Textarea value={notes} onChange={(e) => setNotes(e.target.value)} />

      <Box display="flex" alignItems="center">
        <FormLabel mb="0">Υπήρξε συμβάν;</FormLabel>
        <Switch isChecked={hasIncident} onChange={() => setHasIncident(!hasIncident)} />
        <Text ml={2}>{hasIncident ? "Ναι" : "Όχι"}</Text>
      </Box>

      <FormLabel>Υπογραφή</FormLabel>
      <Box border="1px" borderColor="gray.300" borderRadius="md">
        <SignatureCanvas
          ref={sigCanvasRef}
          penColor="black"
          canvasProps={{ width: 300, height: 100, className: 'sigCanvas' }}
        />
      </Box>
      <Button size="sm" onClick={handleClear}>Καθαρισμός</Button>

      <Button colorScheme="blue" onClick={handleSubmit}>Καταχώρηση</Button>
    </VStack>
  );
};

export default VehiclePickupForm;
