import {
  Box,
  Button,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Heading,
  IconButton,
  useDisclosure,
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalCloseButton,
} from "@chakra-ui/react";
import { AddIcon, DeleteIcon, EditIcon } from "@chakra-ui/icons";
import { useEffect, useState } from "react";
import axios from "../utils/axios";
import VehicleForm from "./VehicleForm";
import DocumentUpload from "./DocumentUpload";

interface Vehicle {
  id: number;
  plate: string;
  brand: string;
  model: string;
  year: number;
  status: string;
  notes?: string;
  branch_id: number | null;
  created_at: string;
}

export default function VehicleManagement() {
  const [vehicles, setVehicles] = useState<Vehicle[]>([]);
  const [editingVehicle, setEditingVehicle] = useState<Vehicle | null>(null);
  const [selectedVehicleId, setSelectedVehicleId] = useState<number | null>(null);

  const {
    isOpen: isFormOpen,
    onOpen: onFormOpen,
    onClose: onFormClose,
  } = useDisclosure();

  const {
    isOpen: isDocsOpen,
    onOpen: onDocsOpen,
    onClose: onDocsClose,
  } = useDisclosure();

  const fetchVehicles = async () => {
    try {
      const res = await axios.get("/vehicles");
      setVehicles(res.data);
    } catch (err) {
      console.error("❌ Error fetching vehicles:", err);
    }
  };

  useEffect(() => {
    fetchVehicles();
  }, []);

  const handleDelete = async (id: number) => {
    try {
      await axios.delete(`/vehicles/${id}`);
      fetchVehicles();
    } catch (err) {
      console.error("❌ Error deleting vehicle:", err);
    }
  };

  const handleEdit = (vehicle: Vehicle) => {
    setEditingVehicle(vehicle);
    onFormOpen();
  };

  const handleDocs = (id: number) => {
    setSelectedVehicleId(id);
    onDocsOpen();
  };

  return (
    <Box p={4}>
      <Heading size="lg" mb={4}>
        Διαχείριση Οχημάτων
      </Heading>

      <Button
        leftIcon={<AddIcon />}
        colorScheme="green"
        mb={4}
        width={{ base: "100%", sm: "auto" }}
        onClick={() => {
          setEditingVehicle(null);
          onFormOpen();
        }}
      >
        Προσθήκη Οχήματος
      </Button>

      <Box overflowX="auto">
        <Table variant="simple">
          <Thead>
            <Tr>
              <Th>Πινακίδα</Th>
              <Th>Μάρκα</Th>
              <Th>Μοντέλο</Th>
              <Th>Έτος</Th>
              <Th>Κατάσταση</Th>
              <Th>Σημειώσεις</Th>
              <Th>Υποκατάστημα</Th>
              <Th>Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {vehicles.map((v) => (
              <Tr key={v.id}>
                <Td>{v.plate}</Td>
                <Td>{v.brand}</Td>
                <Td>{v.model}</Td>
                <Td>{v.year}</Td>
                <Td>{v.status}</Td>
                <Td>{v.notes || "-"}</Td>
                <Td>{v.branch_id === null ? "Διοίκηση" : v.branch_id}</Td>
                <Td>
                  <IconButton
                    icon={<EditIcon />}
                    size="sm"
                    mr={2}
                    onClick={() => handleEdit(v)}
                    aria-label="Edit"
                  />
                  <IconButton
                    icon={<DeleteIcon />}
                    size="sm"
                    mr={2}
                    colorScheme="red"
                    onClick={() => handleDelete(v.id)}
                    aria-label="Delete"
                  />
                  <Button
                    size="sm"
                    colorScheme="blue"
                    onClick={() => handleDocs(v.id)}
                  >
                    📄 Έγγραφα
                  </Button>
                </Td>
              </Tr>
            ))}
          </Tbody>
        </Table>
      </Box>

      {/* Vehicle Form Modal */}
      <VehicleForm
        isOpen={isFormOpen}
        onClose={onFormClose}
        refresh={fetchVehicles}
        vehicle={editingVehicle}
      />

      {/* Document Upload Modal */}
      <Modal isOpen={isDocsOpen} onClose={onDocsClose} size="lg">
        <ModalOverlay />
        <ModalContent>
          <ModalHeader>Διαχείριση Εγγράφων Οχήματος</ModalHeader>
          <ModalCloseButton />
          <ModalBody>
            {selectedVehicleId && (
              <DocumentUpload vehicleId={selectedVehicleId} />
            )}
          </ModalBody>
        </ModalContent>
      </Modal>
    </Box>
  );
}
