import {
  Modal,
  ModalOverlay,
  ModalContent,
  ModalHeader,
  ModalBody,
  ModalFooter,
  ModalCloseButton,
  Button,
  FormControl,
  FormLabel,
  Input,
  Select,
  Textarea,
  useToast,
  VStack,
} from "@chakra-ui/react";
import { useEffect, useState } from "react";
import axios from "../utils/axios";

interface VehicleFormProps {
  isOpen: boolean;
  onClose: () => void;
  refresh: () => void;
  vehicle?: any;
}

export default function VehicleForm({
  isOpen,
  onClose,
  refresh,
  vehicle,
}: VehicleFormProps) {
  const toast = useToast();

  const [plate, setPlate] = useState("");
  const [brand, setBrand] = useState("");
  const [model, setModel] = useState("");
  const [year, setYear] = useState<number | "">("");
  const [status, setStatus] = useState("active");
  const [notes, setNotes] = useState("");
  const [branchId, setBranchId] = useState<string>("");

  const [branches, setBranches] = useState([]);

  useEffect(() => {
    if (vehicle) {
      setPlate(vehicle.plate);
      setBrand(vehicle.brand ?? "");
      setModel(vehicle.model ?? "");
      setYear(vehicle.year ?? "");
      setStatus(vehicle.status ?? "active");
      setNotes(vehicle.notes ?? "");
      setBranchId(vehicle.branch_id ?? "");
    } else {
      setPlate("");
      setBrand("");
      setModel("");
      setYear("");
      setStatus("active");
      setNotes("");
      setBranchId("");
    }
  }, [vehicle]);

  useEffect(() => {
    const fetchBranches = async () => {
      try {
        const res = await axios.get("/branches");
        setBranches(res.data);
      } catch (err) {
        console.error("❌ Failed to load branches", err);
      }
    };
    fetchBranches();
  }, []);

  const handleSubmit = async () => {
    try {
      if (!plate) {
        toast({
          title: "Η πινακίδα είναι υποχρεωτική.",
          status: "error",
          duration: 3000,
          isClosable: true,
        });
        return;
      }

      const payload = {
        plate,
        brand,
        model,
        year: year === "" ? null : year,
        status,
        notes,
        branch_id: branchId === "" ? null : branchId,
      };

      if (vehicle) {
        await axios.put(`/vehicles/${vehicle.id}`, payload);
        toast({ title: "Το όχημα ενημερώθηκε.", status: "success" });
      } else {
        await axios.post("/vehicles", payload);
        toast({ title: "Το όχημα προστέθηκε.", status: "success" });
      }

      onClose();
      refresh();
    } catch (err) {
      console.error("❌ Error submitting vehicle", err);
      toast({ title: "Σφάλμα κατά την αποθήκευση.", status: "error" });
    }
  };

  return (
    <Modal isOpen={isOpen} onClose={onClose} size="lg" isCentered>
      <ModalOverlay />
      <ModalContent>
        <ModalHeader>{vehicle ? "Επεξεργασία Οχήματος" : "Νέο Όχημα"}</ModalHeader>
        <ModalCloseButton />
        <ModalBody>
          <VStack spacing={4} width="100%">
            <FormControl isRequired>
              <FormLabel>Πινακίδα</FormLabel>
              <Input value={plate} onChange={(e) => setPlate(e.target.value)} width="100%" />
            </FormControl>
            <FormControl>
              <FormLabel>Μάρκα</FormLabel>
              <Input value={brand} onChange={(e) => setBrand(e.target.value)} width="100%" />
            </FormControl>
            <FormControl>
              <FormLabel>Μοντέλο</FormLabel>
              <Input value={model} onChange={(e) => setModel(e.target.value)} width="100%" />
            </FormControl>
            <FormControl>
              <FormLabel>Έτος</FormLabel>
              <Input
                type="number"
                value={year}
                onChange={(e) => setYear(Number(e.target.value))}
                width="100%"
              />
            </FormControl>
            <FormControl>
              <FormLabel>Κατάσταση</FormLabel>
              <Select value={status} onChange={(e) => setStatus(e.target.value)} width="100%">
                <option value="active">Ενεργό</option>
                <option value="inactive">Ανενεργό</option>
                <option value="maintenance">Συντήρηση</option>
              </Select>
            </FormControl>
            <FormControl>
              <FormLabel>Υποκατάστημα</FormLabel>
              <Select
                placeholder="Διοίκηση"
                value={branchId}
                onChange={(e) => setBranchId(e.target.value)}
                width="100%"
              >
                {branches.map((b: any) => (
                  <option key={b.id} value={b.id}>
                    {b.name}
                  </option>
                ))}
              </Select>
            </FormControl>
            <FormControl>
              <FormLabel>Σημειώσεις</FormLabel>
              <Textarea value={notes} onChange={(e) => setNotes(e.target.value)} width="100%" />
            </FormControl>
          </VStack>
        </ModalBody>
        <ModalFooter>
          <Button onClick={onClose} mr={3} width={{ base: "100%", sm: "auto" }}>
            Ακύρωση
          </Button>
          <Button colorScheme="blue" onClick={handleSubmit} width={{ base: "100%", sm: "auto" }}>
            Αποθήκευση
          </Button>
        </ModalFooter>
      </ModalContent>
    </Modal>
  );
}
