import React, { useState } from "react";
import {
  Box,
  Button,
  Input,
  Textarea,
  FormLabel,
  Switch,
  VStack,
  Text,
} from "@chakra-ui/react";
import axios from "../utils/axios";

interface VehicleEditFormProps {
  entry: any;
  mode: "pickup" | "return";
  onSuccess: () => void;
}

const VehicleEditForm: React.FC<VehicleEditFormProps> = ({ entry, mode, onSuccess }) => {
  const isPickup = mode === "pickup";

  const [time, setTime] = useState(
    (isPickup ? entry.pickup_time : entry.return_time)?.slice(0, 16) || ""
  );
  const [km, setKm] = useState(isPickup ? entry.start_km : entry.end_km || "");
  const [notes, setNotes] = useState(entry.notes || "");
  const [hasIncident, setHasIncident] = useState(entry.has_incident);

  const handleSubmit = async () => {
    try {
      const payload =
        isPickup
          ? {
              pickup_time: time,
              start_km: km,
              notes,
              has_incident: hasIncident,
            }
          : {
              return_time: time,
              end_km: km,
              notes,
              has_incident: hasIncident,
            };

      const url = isPickup
        ? `/vehicle-logbook/${entry.id}`
        : `/vehicle-logbook/${entry.id}/return`;

      await axios.put(url, payload);

      alert("Η εγγραφή ενημερώθηκε.");
      onSuccess();
    } catch (err) {
      console.error(err);
      alert("Σφάλμα ενημέρωσης.");
    }
  };

  const handleDelete = async () => {
    if (!window.confirm("Είστε σίγουροι για τη διαγραφή της εγγραφής;")) return;
    try {
      await axios.delete(`/vehicle-logbook/${entry.id}`);
      alert("Η εγγραφή διαγράφηκε.");
      onSuccess();
    } catch (err) {
      console.error(err);
      alert("Σφάλμα διαγραφής.");
    }
  };

  return (
    <VStack spacing={4} align="stretch" width="100%">
      <FormLabel>
        {isPickup ? "Ημερομηνία/Ώρα Παραλαβής" : "Ημερομηνία/Ώρα Επιστροφής"}
      </FormLabel>
      <Input
        type="datetime-local"
        value={time}
        onChange={(e) => setTime(e.target.value)}
        width="100%"
      />

      <FormLabel>
        {isPickup ? "Χιλιόμετρα εκκίνησης" : "Χιλιόμετρα επιστροφής"}
      </FormLabel>
      <Input
        type="number"
        value={km}
        onChange={(e) => setKm(e.target.value)}
        width="100%"
      />

      <FormLabel>Παρατηρήσεις</FormLabel>
      <Textarea
        value={notes}
        onChange={(e) => setNotes(e.target.value)}
        width="100%"
      />

      <Box display="flex" alignItems="center">
        <FormLabel mb="0">Υπήρξε συμβάν;</FormLabel>
        <Switch isChecked={hasIncident} onChange={() => setHasIncident(!hasIncident)} />
        <Text ml={2}>{hasIncident ? "Ναι" : "Όχι"}</Text>
      </Box>

      <Box display="flex" flexDirection={{ base: "column", sm: "row" }} gap={3}>
        <Button colorScheme="red" onClick={handleDelete} width={{ base: "100%", sm: "auto" }}>
          Διαγραφή
        </Button>
        <Button colorScheme="yellow" onClick={handleSubmit} width={{ base: "100%", sm: "auto" }}>
          Αποθήκευση
        </Button>
      </Box>
    </VStack>
  );
};

export default VehicleEditForm;
