import { useEffect, useState } from "react";
import {
  Box,
  Heading,
  VStack,
  SimpleGrid,
  FormControl,
  FormLabel,
  NumberInput,
  NumberInputField,
  Select,
  Button,
  Text,
  Divider,
  HStack,
  useToast,
  Spinner,
} from "@chakra-ui/react";
import { useNavigate, useParams } from "react-router-dom";
import api from "../utils/axios";

export default function PayrollForm() {
  const [employees, setEmployees] = useState<any[]>([]);
  const [form, setForm] = useState<any>({
    employee_id: "",
    period: "ΙΑΝΟΥΑΡΙΟΣ",
    year: new Date().getFullYear(),
    base_salary: 0,
    christmas_bonus: 0,
    easter_bonus: 0,
    leave_allowance: 0,
    leave_compensation: 0,
    severance_pay: 0,
    food_card_amount: 0,
    bonus_total: 0,
    overtime_total: 0,
    paid_amount: 0,
  });
  const [loading, setLoading] = useState(false);
  const [loadingData, setLoadingData] = useState(true);
  const { id } = useParams();
  const toast = useToast();
  const navigate = useNavigate();

  const handleChange = (field: string, value: any) => {
    // μετατροπή σε δεκαδικό αριθμό
    const parsed = parseFloat(value.toString().replace(",", "."));
    setForm({ ...form, [field]: isNaN(parsed) ? 0 : parsed });
  };

  // === Load employees ===
  useEffect(() => {
    const fetchEmployees = async () => {
      try {
        const res = await api.get("/employees");
        setEmployees(res.data);
      } catch {
        toast({ title: "Σφάλμα φόρτωσης υπαλλήλων", status: "error" });
      } finally {
        setLoadingData(false);
      }
    };
    fetchEmployees();
  }, []);

  // === Load payroll (edit mode) ===
  useEffect(() => {
    const fetchPayroll = async () => {
      if (!id) return;
      try {
        const res = await api.get(`/payrolls/details/${id}`);
        setForm(res.data);
      } catch {
        toast({ title: "Σφάλμα φόρτωσης μισθοδοσίας", status: "error" });
      }
    };
    fetchPayroll();
  }, [id]);

  // === Save payroll ===
  const handleSubmit = async () => {
    try {
      setLoading(true);
      if (id) {
        await api.put(`/payrolls/${id}`, form);
        toast({ title: "Η μισθοδοσία ενημερώθηκε", status: "success" });
      } else {
        await api.post("/payrolls", form);
        toast({ title: "Η μισθοδοσία καταχωρήθηκε", status: "success" });
      }
      navigate("/payrolls");
    } catch (err) {
      console.error("❌ Error saving payroll:", err);
      toast({ title: "Σφάλμα αποθήκευσης", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  if (loadingData)
    return (
      <Box textAlign="center" py={20}>
        <Spinner size="xl" />
      </Box>
    );

  const totalDue =
    form.base_salary +
    form.christmas_bonus +
    form.easter_bonus +
    form.leave_allowance +
    form.leave_compensation +
    form.severance_pay +
    form.food_card_amount +
    form.bonus_total +
    form.overtime_total;

  const remaining = totalDue - form.paid_amount;

  const greekMonths = [
    "ΙΑΝΟΥΑΡΙΟΣ",
    "ΦΕΒΡΟΥΑΡΙΟΣ",
    "ΜΑΡΤΙΟΣ",
    "ΑΠΡΙΛΙΟΣ",
    "ΜΑΙΟΣ",
    "ΙΟΥΝΙΟΣ",
    "ΙΟΥΛΙΟΣ",
    "ΑΥΓΟΥΣΤΟΣ",
    "ΣΕΠΤΕΜΒΡΙΟΣ",
    "ΟΚΤΩΒΡΙΟΣ",
    "ΝΟΕΜΒΡΙΟΣ",
    "ΔΕΚΕΜΒΡΙΟΣ",
    "ΔΩΡΟ ΠΑΣΧΑ",
    "ΕΠΙΔΟΜΑ ΑΔΕΙΑΣ",
    "ΔΩΡΟ ΧΡΙΣΤΟΥΓΕΝΝΩΝ",
  ];

  return (
    <Box bg="gray.50" minH="100vh" p={{ base: 4, md: 8 }}>
      <Heading size="lg" mb={6} textAlign="center">
        {id ? "✏️ Επεξεργασία Μισθοδοσίας" : "➕ Νέα Μισθοδοσία"}
      </Heading>

      <VStack spacing={6} align="stretch">
        {/* --- Υπάλληλος & Περίοδος --- */}
        <SimpleGrid columns={{ base: 1, md: 3 }} spacing={4}>
          <FormControl isRequired>
            <FormLabel>Υπάλληλος</FormLabel>
            <Select
              value={form.employee_id}
              onChange={(e) => handleChange("employee_id", e.target.value)}
            >
              <option value="">-- Επιλογή --</option>
              {employees.map((e) => (
                <option key={e.id} value={e.id}>
                  {e.last_name
                    ? `${e.last_name} ${e.first_name}`
                    : e.username || "-"}
                </option>
              ))}
            </Select>
          </FormControl>

          <FormControl>
            <FormLabel>Περίοδος</FormLabel>
            <Select
              value={form.period}
              onChange={(e) => handleChange("period", e.target.value)}
            >
              {greekMonths.map((m) => (
                <option key={m} value={m}>
                  {m}
                </option>
              ))}
            </Select>
          </FormControl>

          <FormControl>
            <FormLabel>Έτος</FormLabel>
            <NumberInput
              min={2020}
              max={2100}
              value={form.year}
              onChange={(_, n) => handleChange("year", n)}
            >
              <NumberInputField />
            </NumberInput>
          </FormControl>
        </SimpleGrid>

        <Divider />

        {/* --- Ποσά --- */}
        <SimpleGrid columns={{ base: 1, md: 3 }} spacing={4}>
          {[
            ["base_salary", "Μισθός"],
            ["food_card_amount", "Κάρτα Σίτισης"],
            ["bonus_total", "Επίδομα / Bonus"],
            ["overtime_total", "Υπερωρίες"],
            ["christmas_bonus", "Δώρο Χριστουγέννων"],
            ["easter_bonus", "Δώρο Πάσχα"],
            ["leave_allowance", "Επίδομα Άδειας"],
            ["leave_compensation", "Αποζημίωση Άδειας"],
            ["severance_pay", "Αποζημίωση Απόλυσης"],
            ["paid_amount", "Πληρωθέν Ποσό"],
          ].map(([key, label]) => (
            <FormControl key={key}>
              <FormLabel>{label}</FormLabel>
              <NumberInput
                step={0.01}
                precision={2}
                min={0}
                value={form[key]}
                clampValueOnBlur={false}
                onChange={(valueString) =>
                  handleChange(key, valueString.replace(",", "."))
                }
              >
                <NumberInputField inputMode="decimal" />
              </NumberInput>
            </FormControl>
          ))}
        </SimpleGrid>

        <Divider />

        {/* --- Σύνολα --- */}
        <Box textAlign="center">
          <Text fontSize="lg" fontWeight="bold">
            Σύνολο: {totalDue.toFixed(2)} € | Πληρωμένο:{" "}
            {form.paid_amount.toFixed(2)} € | Υπόλοιπο:{" "}
            <Text
              as="span"
              color={remaining > 0 ? "red.500" : "green.600"}
              fontWeight="bold"
            >
              {remaining.toFixed(2)} €
            </Text>
          </Text>
        </Box>

        {/* --- Κουμπιά --- */}
        <HStack justify="flex-end" pt={4}>
          <Button
            colorScheme="blue"
            onClick={handleSubmit}
            isLoading={loading}
            isDisabled={!form.employee_id}
          >
            {id ? "Αποθήκευση Αλλαγών" : "Καταχώρηση"}
          </Button>

          <Button
            colorScheme="gray"
            onClick={() => {
              if (window.history.length > 2) navigate(-1);
              else navigate("/payrolls");
            }}
          >
            ⬅ Πίσω
          </Button>
        </HStack>
      </VStack>
    </Box>
  );
}
