import {
  Box,
  Button,
  FormControl,
  FormLabel,
  Input,
  Select,
  Textarea,
  VStack,
  Heading,
  useToast,
} from "@chakra-ui/react";
import { useState, useEffect } from "react";
import { useNavigate, useParams } from "react-router-dom";
import api from "../utils/axios";
import dayjs from "dayjs";
import "dayjs/locale/el";
dayjs.locale("el");

export default function InvoiceForm() {
  const { id } = useParams();
  const navigate = useNavigate();
  const toast = useToast();

  const [customers, setCustomers] = useState<any[]>([]);
  const [invoice, setInvoice] = useState({
    customer_id: "",
    invoice_number: "",
    type: "service_invoice",
    elta_code: "",
    issue_date: dayjs().format("YYYY-MM-DD"),
    due_date: dayjs().add(30, "day").format("YYYY-MM-DD"),
    amount: "",
    notes: "",
  });

  // Φόρτωση λίστας πελατών
  useEffect(() => {
    api
      .get("/customers")
      .then((res) => setCustomers(res.data))
      .catch(() => toast({ title: "Σφάλμα φόρτωσης πελατών", status: "error" }));
  }, []);

  // Αν υπάρχει id → επεξεργασία
  useEffect(() => {
    if (id) {
      api
        .get(`/invoices/${id}`)
        .then((res) => setInvoice(res.data))
        .catch(() =>
          toast({ title: "Σφάλμα φόρτωσης τιμολογίου", status: "error" })
        );
    }
  }, [id]);

  const handleChange = (field: string, value: any) => {
    setInvoice((prev) => ({
      ...prev,
      [field]: value,
      ...(field === "issue_date" && {
        due_date: dayjs(value).add(30, "day").format("YYYY-MM-DD"),
      }),
    }));
  };

  const handleSubmit = async () => {
    try {
      if (!invoice.customer_id || !invoice.invoice_number || !invoice.amount) {
        toast({
          title: "Συμπλήρωσε όλα τα υποχρεωτικά πεδία",
          status: "warning",
        });
        return;
      }

      if (id) {
        await api.put(`/invoices/${id}`, invoice);
        toast({ title: "Το τιμολόγιο ενημερώθηκε", status: "success" });
      } else {
        await api.post("/invoices", invoice);
        toast({ title: "Το τιμολόγιο προστέθηκε", status: "success" });
      }

      navigate("/invoices");
    } catch (err) {
      console.error("❌ Error saving invoice:", err);
      toast({ title: "Αποτυχία αποθήκευσης τιμολογίου", status: "error" });
    }
  };

  return (
    <Box
      p={6}
      bg="white"
      shadow="md"
      rounded="lg"
      maxW="700px"
      mx="auto"
      mt={10}
    >
      <Heading size="md" mb={4}>
        {id ? "✏️ Επεξεργασία Τιμολογίου" : "➕ Νέο Τιμολόγιο"}
      </Heading>

      <VStack spacing={3} align="stretch">
        <FormControl isRequired>
          <FormLabel>Πελάτης</FormLabel>
          <Select
            placeholder="Επιλογή πελάτη"
            value={invoice.customer_id}
            onChange={(e) => handleChange("customer_id", e.target.value)}
          >
            {customers.map((c) => (
              <option key={c.id} value={c.id}>
                {c.code ? `${c.code} - ` : ""}
                {c.name}
              </option>
            ))}
          </Select>
        </FormControl>

        <FormControl isRequired>
          <FormLabel>Αριθμός Τιμολογίου</FormLabel>
          <Input
            value={invoice.invoice_number}
            onChange={(e) => handleChange("invoice_number", e.target.value)}
          />
        </FormControl>

        <FormControl>
          <FormLabel>Κωδικός ΕΛΤΑ</FormLabel>
          <Input
            value={invoice.elta_code}
            onChange={(e) => handleChange("elta_code", e.target.value)}
            placeholder="π.χ. 145681 00 00"
          />
        </FormControl>

        <FormControl isRequired>
          <FormLabel>Τύπος</FormLabel>
          <Select
            value={invoice.type}
            onChange={(e) => handleChange("type", e.target.value)}
          >
            <option value="service_invoice">Τιμολόγιο Παροχής Υπηρεσιών</option>
            <option value="credit_invoice">Πιστωτικό Τιμολόγιο</option>
          </Select>
        </FormControl>

        <FormControl isRequired>
          <FormLabel>Ημερομηνία Έκδοσης</FormLabel>
          <Input
            type="date"
            value={invoice.issue_date}
            onChange={(e) => handleChange("issue_date", e.target.value)}
          />
        </FormControl>

        <FormControl>
          <FormLabel>Ημερομηνία Λήξης</FormLabel>
          <Input type="date" value={invoice.due_date} readOnly />
        </FormControl>

        <FormControl isRequired>
          <FormLabel>Ποσό (€)</FormLabel>
          <Input
            type="number"
            step="0.01"
            value={invoice.amount}
            onChange={(e) => handleChange("amount", e.target.value)}
          />
        </FormControl>

        <FormControl>
          <FormLabel>Σημειώσεις</FormLabel>
          <Textarea
            value={invoice.notes}
            onChange={(e) => handleChange("notes", e.target.value)}
          />
        </FormControl>

        <Button colorScheme="blue" onClick={handleSubmit}>
          💾 Αποθήκευση
        </Button>
        <Button variant="outline" onClick={() => navigate(-1)}>
          ⬅ Πίσω
        </Button>
      </VStack>
    </Box>
  );
}
