import { useEffect, useState } from 'react';
import { getEquipment, deleteEquipment, exportEquipment } from '../utils/equipmentAPI';
import { StatusBadge } from './StatusBadge';
import { Button } from './ui/Button';

export default function EquipmentTable({ onSelect }: { onSelect: (id: string) => void }) {
  const [equipment, setEquipment] = useState<any[]>([]);

  const fetchData = async () => {
    const res = await getEquipment();
    setEquipment(res.data);
  };

  useEffect(() => { fetchData(); }, []);

  const handleExport = async () => {
    const res = await exportEquipment();
    const url = window.URL.createObjectURL(new Blob([res.data]));
    const link = document.createElement('a');
    link.href = url;
    link.download = 'equipment_export.xlsx';
    link.click();
  };

  return (
    <div className="p-4">
      <div className="flex justify-between items-center mb-3">
        <h2 className="text-xl font-bold">Equipment Inventory</h2>
        <div className="space-x-2">
          <Button onClick={handleExport}>Export to Excel</Button>
          <Button onClick={fetchData}>Refresh</Button>
        </div>
      </div>

      <table className="min-w-full bg-white border border-gray-300 rounded-lg">
        <thead>
          <tr className="bg-gray-100 text-sm">
            <th className="p-2 text-left">ID</th>
            <th className="p-2 text-left">Brand</th>
            <th className="p-2 text-left">Model</th>
            <th className="p-2 text-left">Status</th>
            <th className="p-2 text-left">Actions</th>
          </tr>
        </thead>
        <tbody>
          {equipment.map((eq) => (
            <tr key={eq.id} className="border-b hover:bg-gray-50">
              <td className="p-2">{eq.asset_tag}</td>
              <td className="p-2">{eq.brand}</td>
              <td className="p-2">{eq.model}</td>
              <td className="p-2"><StatusBadge status={eq.status_code} /></td>
              <td className="p-2 space-x-2">
                <Button size="sm" onClick={() => onSelect(eq.id)}>View</Button>
                <Button size="sm" variant="danger" onClick={() => deleteEquipment(eq.id).then(fetchData)}>Delete</Button>
              </td>
            </tr>
          ))}
        </tbody>
      </table>
    </div>
  );
}
