import {
  Box,
  Button,
  FormControl,
  FormLabel,
  Heading,
  Input,
  NumberInput,
  NumberInputField,
  Select,
  Textarea,
  VStack,
  HStack,
  useToast,
  Tabs,
  TabList,
  TabPanels,
  Tab,
  TabPanel,
  SimpleGrid,
  Divider,
  Spinner,
  Text,
} from "@chakra-ui/react";
import { useEffect, useState } from "react";
import { useNavigate, useParams } from "react-router-dom";
import api from "../utils/axios";
import DatePicker, { registerLocale } from "react-datepicker";
import { el } from "date-fns/locale";
import "react-datepicker/dist/react-datepicker.css";
import { format } from "date-fns";
import EmployeeFiles from "../components/EmployeeFiles";

registerLocale("el", el);

export default function EmployeeForm() {
  const [users, setUsers] = useState<any[]>([]);
  const [branches, setBranches] = useState<any[]>([]);
  const [form, setForm] = useState<any>({
    user_id: "",
    branch_id: "",
    first_name: "",
    last_name: "",
    position: "",
    department: "",
    hire_date: new Date(),
    termination_date: null,
    salary: 0,
    employment_type: "full_time",
    employment_status: "active",
    iban: "",
    afm: "",
    amka: "",
    phone: "",
    address: "",
    emergency_contact: "",
    notes: "",
  });

  const [loading, setLoading] = useState(false);
  const [loadingData, setLoadingData] = useState(true);
  const { id } = useParams();
  const toast = useToast();
  const navigate = useNavigate();

  useEffect(() => {
    const fetchMeta = async () => {
      const [u, b] = await Promise.all([api.get("/users"), api.get("/branches")]);
      setUsers(u.data);
      setBranches(b.data);
    };
    fetchMeta();
  }, []);

  useEffect(() => {
    if (id) {
      api.get(`/employees/${id}`).then((res) => {
        const e = res.data;
        setForm({
          user_id: e.user_id || "",
          branch_id: e.branch_id || "",
          first_name: e.first_name || "",
          last_name: e.last_name || "",
          position: e.position || "",
          department: e.department || "",
          hire_date: e.hire_date ? new Date(e.hire_date) : new Date(),
          termination_date: e.termination_date ? new Date(e.termination_date) : null,
          salary: e.salary || 0,
          employment_type: e.employment_type || "full_time",
          employment_status: e.employment_status || "active",
          iban: e.iban || "",
          afm: e.afm || "",
          amka: e.amka || "",
          phone: e.phone || "",
          address: e.address || "",
          emergency_contact: e.emergency_contact || "",
          notes: e.notes || "",
        });
        setLoadingData(false);
      });
    } else {
      setLoadingData(false);
    }
  }, [id]);

  const handleChange = (field: string, value: any) => {
    setForm({ ...form, [field]: value });
  };

  const handleSubmit = async () => {
    try {
      setLoading(true);
      const payload = {
        ...form,
        hire_date: form.hire_date ? format(form.hire_date, "yyyy-MM-dd") : null,
        termination_date: form.termination_date ? format(form.termination_date, "yyyy-MM-dd") : null,
      };

      if (id) {
        await api.put(`/employees/${id}`, payload);
        toast({ title: "Τα στοιχεία ενημερώθηκαν", status: "success" });
      } else {
        await api.post("/employees", payload);
        toast({ title: "Ο υπάλληλος προστέθηκε", status: "success" });
      }

      navigate("/employees");
    } catch (err) {
      console.error("❌ Error saving employee:", err);
      toast({ title: "Σφάλμα κατά την αποθήκευση", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  if (loadingData)
    return (
      <Box textAlign="center" py={20}>
        <Spinner size="xl" />
      </Box>
    );

  return (
    <Box bg="gray.50" minH="100vh" p={{ base: 4, md: 8 }}>
      <Heading size="lg" mb={6} textAlign="center">
        {id ? "Καρτέλα Υπαλλήλου" : "Προσθήκη Νέου Υπαλλήλου"}
      </Heading>

      <Tabs variant="enclosed-colored" colorScheme="blue" isFitted>
        <TabList mb="1em">
          <Tab fontWeight="bold">🧍 Στοιχεία Υπαλλήλου</Tab>
          <Tab fontWeight="bold" isDisabled={!id}>
            📎 Αρχεία & Συμβάσεις
          </Tab>
        </TabList>

        <TabPanels bg="white" p={6} rounded="xl" shadow="md" minH="80vh">
          {/* --- TAB 1 --- */}
          <TabPanel>
            <VStack spacing={6} align="stretch">
              <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4}>
                <FormControl isRequired>
                  <FormLabel>Συνδεδεμένος Χρήστης</FormLabel>
                  <Select
                    value={form.user_id}
                    onChange={(e) => handleChange("user_id", e.target.value)}
                  >
                    <option value="">-- Επιλογή --</option>
                    {users.map((u) => (
                      <option key={u.id} value={u.id}>
                        {u.username}
                      </option>
                    ))}
                  </Select>
                </FormControl>

                <FormControl>
                  <FormLabel>Υποκατάστημα</FormLabel>
                  <Select
                    value={form.branch_id}
                    onChange={(e) => handleChange("branch_id", e.target.value)}
                  >
                    <option value="">-- Επιλογή --</option>
                    {branches.map((b) => (
                      <option key={b.id} value={b.id}>
                        {b.name}
                      </option>
                    ))}
                  </Select>
                </FormControl>

                <FormControl isRequired>
                  <FormLabel>Όνομα</FormLabel>
                  <Input
                    value={form.first_name}
                    onChange={(e) => handleChange("first_name", e.target.value)}
                    placeholder="Π.χ. Γιάννης"
                  />
                </FormControl>

                <FormControl isRequired>
                  <FormLabel>Επώνυμο</FormLabel>
                  <Input
                    value={form.last_name}
                    onChange={(e) => handleChange("last_name", e.target.value)}
                    placeholder="Π.χ. Μπουρτζινάκος"
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Θέση</FormLabel>
                  <Input
                    value={form.position}
                    onChange={(e) => handleChange("position", e.target.value)}
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Τμήμα</FormLabel>
                  <Input
                    value={form.department}
                    onChange={(e) => handleChange("department", e.target.value)}
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Κατάσταση Εργασίας</FormLabel>
                  <Select
                    value={form.employment_status}
                    onChange={(e) => handleChange("employment_status", e.target.value)}
                  >
                    <option value="active">Ενεργός</option>
                    <option value="on_leave">Σε Άδεια</option>
                    <option value="inactive">Ανενεργός</option>
                    <option value="terminated">Αποχωρήσας</option>
                  </Select>
                </FormControl>

                <FormControl>
                  <FormLabel>Τύπος Εργασίας</FormLabel>
                  <Select
                    value={form.employment_type}
                    onChange={(e) => handleChange("employment_type", e.target.value)}
                  >
                    <option value="full_time">Πλήρης Απασχόληση</option>
                    <option value="part_time">Μερική</option>
                    <option value="contract">Σύμβαση</option>
                    <option value="intern">Πρακτική</option>
                  </Select>
                </FormControl>
              </SimpleGrid>

              <Divider />

              {/* --- Δεύτερο Block --- */}
              <SimpleGrid columns={{ base: 1, md: 2 }} spacing={4}>
                <FormControl>
                  <FormLabel>Ημερομηνία Πρόσληψης</FormLabel>
                  <DatePicker
                    selected={form.hire_date}
                    onChange={(date) => handleChange("hire_date", date)}
                    locale="el"
                    dateFormat="dd/MM/yyyy"
                    className="chakra-input css-1c6xsvs"
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Ημερομηνία Λήξης</FormLabel>
                  <DatePicker
                    selected={form.termination_date}
                    onChange={(date) => handleChange("termination_date", date)}
                    locale="el"
                    dateFormat="dd/MM/yyyy"
                    isClearable
                    placeholderText="-- Επίλεξε ημερομηνία --"
                    className="chakra-input css-1c6xsvs"
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Μισθός (€)</FormLabel>
                  <NumberInput
                    value={form.salary}
                    onChange={(_, n) => handleChange("salary", n)}
                    precision={2}
                  >
                    <NumberInputField />
                  </NumberInput>
                </FormControl>

                <FormControl>
                  <FormLabel>ΙΒΑΝ</FormLabel>
                  <Input value={form.iban} onChange={(e) => handleChange("iban", e.target.value)} />
                </FormControl>

                <FormControl>
                  <FormLabel>Α.Φ.Μ.</FormLabel>
                  <Input value={form.afm} onChange={(e) => handleChange("afm", e.target.value)} />
                </FormControl>

                <FormControl>
                  <FormLabel>Α.Μ.Κ.Α.</FormLabel>
                  <Input value={form.amka} onChange={(e) => handleChange("amka", e.target.value)} />
                </FormControl>

                <FormControl>
                  <FormLabel>Τηλέφωνο</FormLabel>
                  <Input
                    value={form.phone}
                    onChange={(e) => handleChange("phone", e.target.value)}
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Διεύθυνση</FormLabel>
                  <Input
                    value={form.address}
                    onChange={(e) => handleChange("address", e.target.value)}
                  />
                </FormControl>

                <FormControl>
                  <FormLabel>Επικοινωνία Έκτακτης Ανάγκης</FormLabel>
                  <Input
                    value={form.emergency_contact}
                    onChange={(e) => handleChange("emergency_contact", e.target.value)}
                  />
                </FormControl>
              </SimpleGrid>

              <FormControl>
                <FormLabel>Σημειώσεις</FormLabel>
                <Textarea
                  value={form.notes}
                  onChange={(e) => handleChange("notes", e.target.value)}
                />
              </FormControl>

              <HStack justify="flex-end" pt={4}>
                <Button
                  colorScheme="blue"
                  onClick={handleSubmit}
                  isLoading={loading}
                  isDisabled={!form.user_id || !form.first_name || !form.last_name}
                >
                  {id ? "Αποθήκευση Αλλαγών" : "Καταχώρηση Υπαλλήλου"}
                </Button>

                <Button
                  colorScheme="gray"
                  onClick={() => {
                    if (window.history.length > 2) navigate(-1);
                    else navigate("/employees");
                  }}
                >
                  ⬅ Πίσω
                </Button>
              </HStack>
            </VStack>
          </TabPanel>

          {/* --- TAB 2 --- */}
          <TabPanel>
            {id ? (
              <EmployeeFiles employeeId={Number(id)} />
            ) : (
              <Text color="gray.500">
                Αποθήκευσε πρώτα τον υπάλληλο για να ανεβάσεις αρχεία.
              </Text>
            )}
          </TabPanel>
        </TabPanels>
      </Tabs>
    </Box>
  );
}
