import { useEffect, useState } from "react";
import api from "../utils/axios";
import {
  Box,
  Button,
  Input,
  Table,
  Thead,
  Tr,
  Th,
  Td,
  Tbody,
  Text,
  HStack,
  Select,
  Spinner,
  useToast,
  Image,
} from "@chakra-ui/react";

// ✅ Ορισμός backend URL (παίρνει από .env ή default σε localhost:5000)
const API_URL = import.meta.env.VITE_API_URL || "http://localhost:5000";

export default function EmployeeFiles({ employeeId }: { employeeId: number }) {
  const [files, setFiles] = useState<any[]>([]);
  const [file, setFile] = useState<File | null>(null);
  const [fileType, setFileType] = useState("other");
  const [loading, setLoading] = useState(false);
  const toast = useToast();

  const fetchFiles = async () => {
    try {
      setLoading(true);
      const res = await api.get(`/employee-files/${employeeId}`);
      setFiles(res.data);
    } catch (err) {
      toast({ title: "Σφάλμα φόρτωσης αρχείων", status: "error" });
    } finally {
      setLoading(false);
    }
  };

  const handleUpload = async () => {
    if (!file) return toast({ title: "Επιλέξτε αρχείο", status: "warning" });

    const formData = new FormData();
    formData.append("file", file);
    formData.append("employee_id", employeeId.toString());
    formData.append("file_type", fileType);

    try {
      await api.post("/employee-files", formData, {
        headers: { "Content-Type": "multipart/form-data" },
      });
      toast({ title: "Το αρχείο ανέβηκε επιτυχώς", status: "success" });
      setFile(null);
      fetchFiles();
    } catch (err) {
      toast({ title: "Σφάλμα κατά το ανέβασμα", status: "error" });
    }
  };

  const handleDelete = async (id: number) => {
    if (!window.confirm("Να διαγραφεί το αρχείο;")) return;
    try {
      await api.delete(`/employee-files/${id}`);
      fetchFiles();
      toast({ title: "Το αρχείο διαγράφηκε", status: "info" });
    } catch (err) {
      toast({ title: "Σφάλμα διαγραφής", status: "error" });
    }
  };

  useEffect(() => {
    if (employeeId) fetchFiles();
  }, [employeeId]);

  const getFileUrl = (filename: string) =>
    `${API_URL}/uploads/employee_files/${filename}`;

  const isImage = (filename: string) => /\.(jpg|jpeg|png|gif)$/i.test(filename);
  const isPDF = (filename: string) => /\.pdf$/i.test(filename);

  return (
    <Box mt={6} p={5} bg="gray.100" rounded="lg" border="1px solid #ddd">
      <Text fontWeight="bold" fontSize="lg" mb={4}>
        📎 Αρχεία & Συμβάσεις
      </Text>

      <HStack spacing={3} mb={5}>
        <Input
          type="file"
          onChange={(e) => setFile(e.target.files?.[0] || null)}
          width="auto"
        />
        <Select
          value={fileType}
          onChange={(e) => setFileType(e.target.value)}
          width="200px"
        >
          <option value="contract">Σύμβαση</option>
          <option value="amendment">Τροποποίηση</option>
          <option value="certificate">Πιστοποιητικό</option>
          <option value="id">Ταυτότητα</option>
          <option value="other">Άλλο</option>
        </Select>
        <Button
          colorScheme="blue"
          onClick={handleUpload}
          isDisabled={!file}
          isLoading={loading}
        >
          Ανέβασμα
        </Button>
      </HStack>

      {loading ? (
        <Spinner />
      ) : files.length ? (
        <Table size="sm" variant="striped">
          <Thead>
            <Tr>
              <Th>Προεπισκόπηση</Th>
              <Th>Αρχείο</Th>
              <Th>Τύπος</Th>
              <Th>Ημερομηνία</Th>
              <Th textAlign="center">Ενέργειες</Th>
            </Tr>
          </Thead>
          <Tbody>
            {files.map((f) => (
              <Tr key={f.id}>
                <Td>
                  {isImage(f.filename) ? (
                    <Image
                      src={getFileUrl(f.filename)}
                      alt={f.filename}
                      boxSize="60px"
                      objectFit="cover"
                      borderRadius="md"
                    />
                  ) : isPDF(f.filename) ? (
                    <a
                      href={getFileUrl(f.filename)}
                      target="_blank"
                      rel="noopener noreferrer"
                      style={{ color: "blue" }}
                    >
                      📄 Προβολή PDF
                    </a>
                  ) : (
                    <Text color="gray.500">--</Text>
                  )}
                </Td>
                <Td>
                  <a
                    href={getFileUrl(f.filename)}
                    target="_blank"
                    rel="noopener noreferrer"
                    style={{ color: "blue", textDecoration: "underline" }}
                  >
                    {f.filename}
                  </a>
                </Td>
                <Td>{f.file_type}</Td>
                <Td>{new Date(f.uploaded_at).toLocaleDateString("el-GR")}</Td>
                <Td textAlign="center">
                  <Button
                    size="xs"
                    colorScheme="red"
                    onClick={() => handleDelete(f.id)}
                  >
                    Διαγραφή
                  </Button>
                </Td>
              </Tr>
            ))}
          </Tbody>
        </Table>
      ) : (
        <Text color="gray.500">Δεν υπάρχουν ανεβασμένα αρχεία.</Text>
      )}
    </Box>
  );
}
