import {
  Box,
  Button,
  Input,
  VStack,
  List,
  ListItem,
  HStack,
  Text,
  IconButton,
  useToast,
} from "@chakra-ui/react";
import { DeleteIcon, DownloadIcon } from "@chakra-ui/icons";
import axios from "../utils/axios";
import { useEffect, useState } from "react";

interface Document {
  id: number;
  vehicle_id: number;
  filename: string;
  filepath: string;
  created_at: string;
}

export default function DocumentUpload({ vehicleId }: { vehicleId: number }) {
  const [file, setFile] = useState<File | null>(null);
  const [documents, setDocuments] = useState<Document[]>([]);
  const toast = useToast();

  // === Φόρτωση λίστας εγγράφων ===
  const fetchDocuments = async () => {
    try {
      const res = await axios.get(`/vehicles/${vehicleId}/documents`);
      setDocuments(res.data);
    } catch (err) {
      console.error("❌ Fetch error:", err);
      toast({
        title: "Σφάλμα φόρτωσης εγγράφων.",
        status: "error",
        duration: 3000,
        isClosable: true,
      });
    }
  };

  // === Ανέβασμα αρχείου ===
  const handleUpload = async () => {
    if (!file) return;
    const formData = new FormData();
    formData.append("file", file); // 👈 πρέπει να είναι "file" όπως περιμένει ο backend

    try {
      await axios.post(`/vehicles/${vehicleId}/documents`, formData, {
        headers: { "Content-Type": "multipart/form-data" },
      });
      toast({
        title: "Επιτυχές ανέβασμα αρχείου.",
        status: "success",
        duration: 2000,
        isClosable: true,
      });
      setFile(null);
      fetchDocuments();
    } catch (err) {
      console.error("❌ Upload error:", err);
      toast({
        title: "Αποτυχία ανεβάσματος.",
        status: "error",
        duration: 3000,
        isClosable: true,
      });
    }
  };

  // === Διαγραφή αρχείου ===
  const handleDelete = async (docId: number) => {
    try {
      await axios.delete(`/vehicles/${vehicleId}/documents/${docId}`);
      toast({
        title: "Το αρχείο διαγράφηκε.",
        status: "info",
        duration: 2000,
        isClosable: true,
      });
      fetchDocuments();
    } catch (err) {
      console.error("❌ Delete error:", err);
      toast({
        title: "Σφάλμα διαγραφής.",
        status: "error",
        duration: 3000,
        isClosable: true,
      });
    }
  };

  // === Λήψη αρχείου ===
  const handleDownload = async (docId: number, filename: string) => {
    try {
      const response = await axios.get(
        `/vehicles/${vehicleId}/documents/${docId}`,
        { responseType: "blob" }
      );
      const blob = new Blob([response.data]);
      const url = window.URL.createObjectURL(blob);

      const link = document.createElement("a");
      link.href = url;
      link.setAttribute("download", filename);
      document.body.appendChild(link);
      link.click();
      link.remove();

      window.URL.revokeObjectURL(url);
    } catch (err) {
      console.error("❌ Download error:", err);
      toast({
        title: "Αποτυχία λήψης αρχείου.",
        status: "error",
        duration: 3000,
        isClosable: true,
      });
    }
  };

  useEffect(() => {
    if (vehicleId) {
      fetchDocuments();
    }
  }, [vehicleId]);

  return (
    <VStack spacing={4} align="stretch" width="100%">
      <Input
        type="file"
        onChange={(e) => setFile(e.target.files?.[0] || null)}
        width="100%"
      />
      <Button
        onClick={handleUpload}
        colorScheme="blue"
        isDisabled={!file}
        width={{ base: "100%", sm: "200px" }}
      >
        Ανεβάστε αρχείο
      </Button>

      <Box overflowX="auto">
        <List spacing={3} minW="100%">
          {documents.length === 0 && (
            <Text color="gray.500" textAlign="center">
              Δεν υπάρχουν ανεβασμένα έγγραφα.
            </Text>
          )}
          {documents.map((doc) => (
            <ListItem key={doc.id}>
              <HStack justify="space-between" flexWrap="wrap">
                <Text>{doc.filename}</Text>
                <HStack>
                  <IconButton
                    icon={<DownloadIcon />}
                    aria-label="Download"
                    size="sm"
                    colorScheme="green"
                    onClick={() => handleDownload(doc.id, doc.filename)}
                  />
                  <IconButton
                    icon={<DeleteIcon />}
                    aria-label="Delete"
                    size="sm"
                    colorScheme="red"
                    onClick={() => handleDelete(doc.id)}
                  />
                </HStack>
              </HStack>
            </ListItem>
          ))}
        </List>
      </Box>
    </VStack>
  );
}
