import { useState } from "react";
import {
  Box,
  Button,
  Input,
  useToast,
  VStack,
  Heading,
  Textarea,
  Text,
  Spinner,
  HStack,
  IconButton,
} from "@chakra-ui/react";
import { ArrowBackIcon } from "@chakra-ui/icons";
import { useNavigate } from "react-router-dom";
import api from "../utils/axios";

export default function CustomerImport() {
  const [file, setFile] = useState<File | null>(null);
  const [result, setResult] = useState<any>(null);
  const [loading, setLoading] = useState(false);
  const toast = useToast();
  const navigate = useNavigate();

  // === Λειτουργία Import ===
  const handleImport = async () => {
    if (!file) {
      toast({ title: "Επίλεξε αρχείο πρώτα.", status: "warning" });
      return;
    }

    const formData = new FormData();
    formData.append("file", file);

    try {
      setLoading(true);
      setResult(null);
      const res = await api.post("/customers/import", formData, {
        headers: { "Content-Type": "multipart/form-data" },
      });

      const { summary, details } = res.data;

      toast({
        title: "✅ Import ολοκληρώθηκε",
        description: `Νέοι: ${summary.inserted} | Ενημερωμένοι: ${summary.updated} | Απέτυχαν: ${summary.failed}`,
        status: summary.failed > 0 ? "warning" : "success",
        duration: 6000,
      });

      setResult({ summary, details });
    } catch (err) {
      console.error(err);
      toast({
        title: "❌ Σφάλμα import",
        description: "Πρόβλημα κατά την εισαγωγή του αρχείου.",
        status: "error",
      });
    } finally {
      setLoading(false);
    }
  };

  // === Εξαγωγή CSV ===
  const exportToCSV = () => {
    if (!result) return;

    const lines = [];
    lines.push("Κατηγορία;Κωδικός;Επωνυμία;Μήνυμα");

    result.details.inserted.forEach((x: any) =>
      lines.push(`Νέα;${x.code};${x.name};Επιτυχής εισαγωγή`)
    );
    result.details.updated.forEach((x: any) =>
      lines.push(`Ενημέρωση;${x.code};${x.name};Επιτυχής ενημέρωση`)
    );
    result.details.failed.forEach((x: any) =>
      lines.push(`Αποτυχία;${x.code};${x.name};${x.error}`)
    );

    const blob = new Blob(["\uFEFF" + lines.join("\n")], {
      type: "text/csv;charset=utf-8;",
    });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = `import_results_${new Date()
      .toISOString()
      .slice(0, 19)
      .replace(/:/g, "-")}.csv`;
    a.click();
  };

  return (
    <Box maxW="8xl" mx="auto" mt={8} bg="gray.50" p={6} rounded="md" shadow="sm">
      {/* === Header με πίσω κουμπί === */}
      <HStack justify="space-between" mb={6}>
        <HStack spacing={3}>
          <IconButton
            aria-label="Πίσω"
            icon={<ArrowBackIcon />}
            colorScheme="red"
            onClick={() => navigate("/customers/list")}
          />
          <Heading size="lg">📤 Εισαγωγή Πελατών ΕΛΤΑ</Heading>
        </HStack>
      </HStack>

      <VStack spacing={4} align="stretch" w="100%">
        <Input
          type="file"
          accept=".xls,.xlsx"
          onChange={(e) => setFile(e.target.files?.[0] || null)}
          bg="white"
        />

        <Button
          colorScheme="teal"
          onClick={handleImport}
          isLoading={loading}
          loadingText="Επεξεργασία..."
        >
          🚀 Εισαγωγή Αρχείου
        </Button>

        {loading && (
          <Box textAlign="center" py={4}>
            <Spinner size="lg" />
            <Text mt={2}>Επεξεργασία αρχείου...</Text>
          </Box>
        )}

        {result && (
          <Box mt={6} bg="white" p={4} rounded="md" shadow="xs">
            <Heading size="md" mb={3}>
              📊 Αποτελέσματα Import
            </Heading>

            <Text mb={2}>
              ✅ Νέοι: <b>{result.summary.inserted}</b> &nbsp;|&nbsp; 🔄 Ενημερωμένοι:{" "}
              <b>{result.summary.updated}</b> &nbsp;|&nbsp; ⚠️ Απέτυχαν:{" "}
              <b>{result.summary.failed}</b>
            </Text>

            <HStack mb={3}>
              <Button colorScheme="blue" onClick={exportToCSV}>
                📥 Εξαγωγή Αποτελεσμάτων σε CSV
              </Button>
            </HStack>

            <Textarea
              readOnly
              value={
                [
                  "=== Νέες εγγραφές ===",
                  ...result.details.inserted.map(
                    (x: any) => `+ ${x.code} — ${x.name}`
                  ),
                  "",
                  "=== Ενημερωμένες εγγραφές ===",
                  ...result.details.updated.map(
                    (x: any) => `↻ ${x.code} — ${x.name}`
                  ),
                  "",
                  result.details.failed.length > 0
                    ? "=== Αποτυχημένες εγγραφές ==="
                    : "",
                  ...result.details.failed.map(
                    (x: any) => `⚠️ ${x.code} — ${x.name} (${x.error})`
                  ),
                ]
                  .filter(Boolean)
                  .join("\n") || "Καμία εγγραφή προς εμφάνιση."
              }
              fontFamily="monospace"
              fontSize="sm"
              rows={20}
            />
          </Box>
        )}
      </VStack>
    </Box>
  );
}
