import { useEffect, useState } from "react";
import {
  Box,
  Heading,
  Table,
  Thead,
  Tbody,
  Tr,
  Th,
  Td,
  Spinner,
  Text,
  Input,
  Flex,
  Button,
  Divider,
} from "@chakra-ui/react";
import api from "../utils/axios";
import { el } from "date-fns/locale";
import DatePicker from "react-datepicker";
import "react-datepicker/dist/react-datepicker.css";


// 🔧 helper για format ημερομηνίας σε ελληνικό format
const formatGreekDate = (dateString: string) => {
  const date = new Date(dateString);
  return date.toLocaleDateString("el-GR", {
    day: "2-digit",
    month: "2-digit",
    year: "numeric",
  });
};

export default function CashOverviewWidget() {
  const today = new Date().toISOString().split("T")[0];
  const [report, setReport] = useState<any>(null);
  const [date, setDate] = useState<string>(today);
  const [loading, setLoading] = useState(false);

  const fetchData = async (selectedDate: string) => {
    try {
      setLoading(true);
      const res = await api.get(`/cash-sessions/reports/daily?date=${selectedDate}`);
      setReport(res.data);
    } catch (err) {
      console.error("Error fetching cash report:", err);
      setReport(null);
    } finally {
      setLoading(false);
    }
  };

  useEffect(() => {
    fetchData(date);
  }, [date]);

  return (
    <Box bg="white" p={4} rounded="xl" shadow="md" border="1px solid" borderColor="gray.100">
      <Heading size="md" mb={4}>
        Ταμείο Ημέρας — {formatGreekDate(date)}
      </Heading>

      {/* Επιλογή ημερομηνίας */}
      <Flex mb={4} gap={3} align="center" wrap="wrap">
  <Text fontWeight="medium">Ημερομηνία:</Text>

  <Box>
    <DatePicker
      selected={new Date(date)}
      onChange={(d: Date) => {
        const iso = d.toISOString().split("T")[0];
        setDate(iso);
      }}
      locale={el}
      dateFormat="dd/MM/yyyy"
      maxDate={new Date()}
      className="chakra-input css-override-datepicker"
    />
  </Box>

  <Button colorScheme="blue" onClick={() => fetchData(date)}>
    Εμφάνιση
  </Button>
</Flex>


      {loading ? (
        <Spinner size="lg" />
      ) : report?.grouped?.length ? (
        <>
          {/* --- Γενικά σύνολα --- */}
          <Text fontWeight="bold" mb={3}>
            Σύνολο Δηλωθέντων: {Number(report.summary.total_declared).toFixed(2)} € |{" "}
            Σύνολο Συστήματος: {Number(report.summary.total_system).toFixed(2)} € | Διαφορά:{" "}
            <Text
              as="span"
              color={Number(report.summary.total_diff) >= 0 ? "green.500" : "red.500"}
              fontWeight="bold"
            >
              {Number(report.summary.total_diff).toFixed(2)} €
            </Text>
          </Text>

          {/* --- Ανά κατάστημα --- */}
          {report.grouped.map((branch: any, idx: number) => (
            <Box key={idx} mb={6}>
              <Heading size="sm" color="blue.600" mb={2}>
                Κατάστημα: {branch.branch_name}
              </Heading>

              <Table variant="striped" size="sm">
                <Thead>
                  <Tr>
                    <Th>Κατηγορία</Th>
                    <Th isNumeric>Σύστημα (€)</Th>
                    <Th isNumeric>Δηλωθέν (€)</Th>
                    <Th isNumeric>Έξοδα (€)</Th>
                    <Th isNumeric>Διαφορά (€)</Th>
                  </Tr>
                </Thead>
                <Tbody>
                  {branch.types.map((type: any, i: number) => (
                    <Tr key={i}>
                      <Td>{type.cash_type}</Td>
                      <Td isNumeric>{type.total_system.toFixed(2)}</Td>
                      <Td isNumeric>{type.total_declared.toFixed(2)}</Td>
                      <Td isNumeric>{(type.total_expenses || 0).toFixed(2)}</Td>
                      <Td
                        isNumeric
                        color={type.total_diff >= 0 ? "green.500" : "red.500"}
                        fontWeight="bold"
                      >
                        {type.total_diff.toFixed(2)}
                      </Td>
                    </Tr>
                  ))}
                  <Tr bg="gray.100" fontWeight="bold">
                    <Td>Σύνολο {branch.branch_name}</Td>
                    <Td isNumeric>{branch.subtotal_system.toFixed(2)}</Td>
                    <Td isNumeric>{branch.subtotal_declared.toFixed(2)}</Td>
                    <Td isNumeric>{branch.subtotal_expenses?.toFixed(2) || "0.00"}</Td>
                    <Td
                      isNumeric
                      color={branch.subtotal_diff >= 0 ? "green.600" : "red.600"}
                    >
                      {branch.subtotal_diff.toFixed(2)}
                    </Td>
                  </Tr>
                </Tbody>
              </Table>

              <Divider my={4} />
            </Box>
          ))}

          {/* --- Τελικό συνολικό --- */}
          <Box mt={4} textAlign="right" fontWeight="bold">
            <Text>
              Συνολικά Δηλωθέντα: {Number(report.summary.total_declared).toFixed(2)} € | Συνολικά
              Συστήματος: {Number(report.summary.total_system).toFixed(2)} € | Διαφορά:{" "}
              <Text
                as="span"
                color={Number(report.summary.total_diff) >= 0 ? "green.600" : "red.600"}
              >
                {Number(report.summary.total_diff).toFixed(2)} €
              </Text>
            </Text>
          </Box>
        </>
      ) : (
        <Text color="gray.400" mt={4}>
          Δεν υπάρχουν καταχωρημένα ταμεία για {formatGreekDate(date)}
        </Text>
      )}
    </Box>
  );
}
