// CashEntryForm.tsx – Πλήρης έκδοση με ελληνικά δεκαδικά, upload εξόδων και καθαρό parsing

import {
  Box,
  Button,
  FormControl,
  FormLabel,
  Heading,
  Input,
  Select,
  SimpleGrid,
  Text,
  VStack,
  useToast,
  Textarea,
  IconButton,
  HStack,
} from "@chakra-ui/react";
import { AddIcon, DeleteIcon } from "@chakra-ui/icons";
import { useEffect, useState } from "react";
import axios from "../utils/axios";
import { useNavigate } from "react-router-dom";
import DatePicker from "react-datepicker";
import { el } from "date-fns/locale";
import "react-datepicker/dist/react-datepicker.css";

const denominationsList = [500, 200, 100, 50, 20, 10, 5, 2, 1, 0.5, 0.2, 0.1, 0.05, 0.02, 0.01];

export default function CashEntryForm({ initialData, onClose, onSuccess, showTotals = true }) {
  const navigate = useNavigate();
  const toast = useToast();

  const [users, setUsers] = useState([]);
  const [branches, setBranches] = useState([]);

  const [form, setForm] = useState({
    user_id: "",
    branch_id: "",
    session_date: "",
    cash_type: "COURIER",
    declared_cash: 0,
    notes: "",
    pos_charges: "",
    pos_cod: "",
    pos_fastpay: "",
    cash_charges: "",
    cash_cod: "",
    cash_fastpay: "",
    denominations: denominationsList.map((v) => ({ value: v, quantity: 0 })),
    expenses: [],
  });

  // ✅ Καθαρισμός & μετατροπή δεκαδικών (δέχεται "," ή ".")
  const parseDecimal = (val) => {
    if (!val) return 0;
    const clean = val.replace(",", ".").replace(/[^0-9.]/g, "");
    return parseFloat(clean) || 0;
  };

  const handleNumericChange = (field, val) => {
    setForm({ ...form, [field]: val });
  };

  useEffect(() => {
    const fetchData = async () => {
      const [u, b] = await Promise.all([axios.get("/users"), axios.get("/branches")]);
      setUsers(u.data);
      setBranches(b.data);
    };
    fetchData();
  }, []);

  useEffect(() => {
    if (initialData) {
      setForm({
        user_id: String(initialData.user_id),
        branch_id: String(initialData.branch_id),
        session_date: initialData.session_date?.substring(0, 10) || "",
        cash_type: initialData.cash_type || "COURIER",
        pos_charges: String(initialData.pos_charges || ""),
        pos_cod: String(initialData.pos_cod || ""),
        pos_fastpay: String(initialData.pos_fastpay || ""),
        cash_charges: String(initialData.cash_charges || ""),
        cash_cod: String(initialData.cash_cod || ""),
        cash_fastpay: String(initialData.cash_fastpay || ""),
        declared_cash: Number(initialData.declared_cash || 0),
        notes: initialData.notes || "",
        denominations: denominationsList.map((v) => {
          const match = initialData.denominations?.find((d) => Number(d.value) === v);
          return { value: v, quantity: match ? Number(match.quantity) : 0 };
        }),
        expenses:
          initialData.expenses?.map((e) => ({
            amount: String(e.amount),
            description: e.description,
            receipt_file: e.receipt_file || null,
            file: null,
          })) || [],
      });
    }
  }, [initialData]);

  // Υπολογισμοί
  const systemCash =
    parseDecimal(form.cash_charges) +
    parseDecimal(form.cash_cod) +
    parseDecimal(form.cash_fastpay);
  const totalDeclared = form.denominations.reduce((sum, d) => sum + d.value * d.quantity, 0);
  const totalExpenses = form.expenses.reduce(
    (sum, e) => sum + (parseDecimal(e.amount) || 0),
    0
  );
  const difference = totalDeclared - systemCash + totalExpenses;

  const handleChangeQty = (index, val) => {
    const newDenoms = [...form.denominations];
    newDenoms[index].quantity = val;
    setForm({ ...form, denominations: newDenoms });
  };

  const handleExpenseChange = (index, field, value) => {
    const updated = [...form.expenses];
    updated[index][field] = value;
    setForm({ ...form, expenses: updated });
  };

  const addExpense = () => {
    setForm({
      ...form,
      expenses: [...form.expenses, { amount: "", description: "", file: null }],
    });
  };

  const removeExpense = (index) => {
    const updated = [...form.expenses];
    updated.splice(index, 1);
    setForm({ ...form, expenses: updated });
  };

  const handleSubmit = async () => {
    try {
      const uploadedExpenses = await Promise.all(
        form.expenses.map(async (e) => {
          if (e.file) {
            const formData = new FormData();
            formData.append("file", e.file);
            const res = await axios.post("/upload/expense", formData, {
              headers: { "Content-Type": "multipart/form-data" },
            });
            return { ...e, receipt_file: res.data.filename };
          }
          return { ...e, receipt_file: e.receipt_file || null };
        })
      );

      const payload = {
        ...form,
        system_cash: systemCash,
        declared_cash: totalDeclared,
        pos_charges: parseDecimal(form.pos_charges),
        pos_cod: parseDecimal(form.pos_cod),
        pos_fastpay: parseDecimal(form.pos_fastpay),
        cash_charges: parseDecimal(form.cash_charges),
        cash_cod: parseDecimal(form.cash_cod),
        cash_fastpay: parseDecimal(form.cash_fastpay),
        denominations: form.denominations.filter((d) => d.quantity > 0),
        expenses: uploadedExpenses
          .filter((e) => parseDecimal(e.amount) > 0 && e.description)
          .map((e) => ({
            ...e,
            amount: parseDecimal(e.amount),
          })),
      };

      if (initialData?.id) {
        await axios.put(`/cash-sessions/${initialData.id}`, payload);
        toast({ title: "Η εγγραφή ενημερώθηκε", status: "success" });
      } else {
        await axios.post("/cash-sessions", payload);
        toast({ title: "Η εγγραφή προστέθηκε", status: "success" });
      }

      if (onSuccess) onSuccess();
    } catch (err) {
      console.error("❌ Submit error:", err);
      toast({ title: "Σφάλμα καταχώρησης", status: "error" });
    }
  };

  const isValid = form.user_id && form.branch_id && form.session_date && form.cash_type;

  return (
    <Box maxW="container.md" mx="auto" p={4} minH="100vh" bg="gray.50">
      <Heading size="md" mb={4} textAlign="center">
        Καταμέτρηση Ταμείου
      </Heading>

      <VStack spacing={4} align="stretch">
        {/* === Επιλογές === */}
        <FormControl isRequired>
          <FormLabel>Χρήστης</FormLabel>
          <Select
            value={form.user_id}
            onChange={(e) => setForm({ ...form, user_id: e.target.value })}
          >
            <option value="">-- Επιλογή --</option>
            {users.map((u) => (
              <option key={u.id} value={u.id}>
                {u.username}
              </option>
            ))}
          </Select>
        </FormControl>

        <FormControl isRequired>
          <FormLabel>Υποκατάστημα</FormLabel>
          <Select
            value={form.branch_id}
            onChange={(e) => setForm({ ...form, branch_id: e.target.value })}
          >
            <option value="">-- Επιλογή --</option>
            {branches.map((b) => (
              <option key={b.id} value={b.id}>
                {b.name}
              </option>
            ))}
          </Select>
        </FormControl>

        {/* === Ημερομηνία === */}
        <FormControl isRequired>
          <FormLabel>Ημερομηνία</FormLabel>
          <DatePicker
            selected={form.session_date ? new Date(form.session_date) : new Date()}
            onChange={(date) => {
              const iso = date.toISOString().split("T")[0];
              setForm({ ...form, session_date: iso });
            }}
            locale={el}
            dateFormat="dd/MM/yyyy"
            maxDate={new Date()}
            className="chakra-input css-datepicker"
            placeholderText="Επιλέξτε ημερομηνία"
          />
        </FormControl>

        {/* === Τύπος Ταμείου === */}
        <FormControl isRequired>
          <FormLabel>Τύπος Ταμείου</FormLabel>
          <Select
            value={form.cash_type}
            onChange={(e) => setForm({ ...form, cash_type: e.target.value })}
          >
            <option value="COURIER">Courier</option>
            <option value="ELTA">ΕΛΤΑ</option>
            <option value="BANK">Τράπεζα</option>
          </Select>
        </FormControl>

        {/* === POS & Μετρητά === */}
        {[
          ["POS Χρεώσεις", "pos_charges"],
          ["POS Αντικαταβολές", "pos_cod"],
          ["POS Ταχυπληρωμές", "pos_fastpay"],
          ["Μετρητά Χρεώσεων", "cash_charges"],
          ["Μετρητά Αντικαταβολών", "cash_cod"],
          ["Μετρητά Ταχυπληρωμών", "cash_fastpay"],
        ].map(([label, field]) => (
          <FormControl key={field}>
            <FormLabel>{label}</FormLabel>
            <Input
              type="text"
              value={form[field]}
              onChange={(e) => handleNumericChange(field, e.target.value)}
              placeholder="π.χ. 150,75"
            />
          </FormControl>
        ))}

        {/* === Καταμέτρηση === */}
        <Box>
          <Heading size="sm" mb={2}>
            Καταμέτρηση Χαρτονομισμάτων/Κερμάτων
          </Heading>
          <SimpleGrid columns={{ base: 2, md: 4 }} spacing={2}>
            {form.denominations.map((d, idx) => (
              <FormControl key={d.value}>
                <FormLabel fontSize="sm">€ {d.value.toFixed(2)}</FormLabel>
                <Input
                  type="number"
                  min={0}
                  value={d.quantity}
                  onChange={(e) => handleChangeQty(idx, parseInt(e.target.value) || 0)}
                />
              </FormControl>
            ))}
          </SimpleGrid>
        </Box>

        {/* === Σύνολα === */}
        {showTotals && (
          <Box>
            <Text fontWeight="bold">Θεωρητικό Ποσό: € {systemCash.toFixed(2)}</Text>
            <Text fontWeight="bold">Πραγματικό Ποσό: € {totalDeclared.toFixed(2)}</Text>
            <Text fontWeight="bold">Σύνολο Εξόδων: € {totalExpenses.toFixed(2)}</Text>
            <Text
              fontWeight="bold"
              color={
                difference < 0 ? "red.500" : difference === 0 ? "blue.500" : "green.600"
              }
            >
              Διαφορά: € {difference.toFixed(2)}
            </Text>
          </Box>
        )}

        {/* === Έξοδα === */}
        <Box>
          <FormLabel mt={4}>Έξοδα (με παραστατικό προαιρετικά)</FormLabel>
          {form.expenses.map((e, idx) => (
            <VStack key={idx} align="stretch" mb={2}>
              <HStack>
                <Input
                  type="text"
                  value={e.amount}
                  onChange={(ev) => handleExpenseChange(idx, "amount", ev.target.value)}
                  placeholder="Ποσό (€)"
                />
                <Input
                  type="file"
                  accept=".pdf,.jpg,.jpeg,.png"
                  onChange={(event) =>
                    handleExpenseChange(idx, "file", event.target.files?.[0] || null)
                  }
                />
                {e.receipt_file && (
                  <Text fontSize="sm" color="gray.600">
                    <a
                      href={`/uploads/expenses/${e.receipt_file}`}
                      target="_blank"
                      rel="noopener noreferrer"
                    >
                      {e.receipt_file}
                    </a>
                  </Text>
                )}
                <IconButton
                  icon={<DeleteIcon />}
                  onClick={() => removeExpense(idx)}
                  aria-label="Διαγραφή"
                />
              </HStack>
              <Textarea
                value={e.description}
                onChange={(ev) => handleExpenseChange(idx, "description", ev.target.value)}
                placeholder="Αιτιολογία"
              />
            </VStack>
          ))}
          <Button leftIcon={<AddIcon />} onClick={addExpense} size="sm" mt={2}>
            Προσθήκη Εξόδου
          </Button>
        </Box>

        {/* === Κουμπιά === */}
        <Button colorScheme="blue" onClick={handleSubmit} isDisabled={!isValid}>
          {initialData?.id ? "Αποθήκευση Αλλαγών" : "Υποβολή Καταμέτρησης"}
        </Button>

        {onClose && (
          <Button colorScheme="gray" onClick={onClose}>
            ⬅ Πίσω
          </Button>
        )}
      </VStack>
    </Box>
  );
}
