import express from "express";
import { uploadSignature } from "../utils/upload.js";
import pool from "../db.js";

const router = express.Router();

// 📌 POST /api/vehicle-logbook/:id/signature
router.post("/:id/signature", uploadSignature.single("file"), async (req, res) => {
  const logbook_id = req.params.id;
  const { type } = req.body; // pickup ή return
  const file_path = req.file?.path;

  if (!file_path || !["pickup", "return"].includes(type)) {
    return res.status(400).json({ error: "Missing file or invalid type." });
  }

  try {
    await pool.execute(
      `INSERT INTO vehicle_logbook_signatures (logbook_id, type, file_path) VALUES (?, ?, ?)`,
      [logbook_id, type, file_path]
    );
    res.status(201).json({ message: "Υπογραφή καταχωρήθηκε." });
  } catch (err) {
    console.error("❌ Signature upload error:", err);
    res.status(500).json({ error: "Σφάλμα αποθήκευσης υπογραφής." });
  }
});

// 📌 GET /api/vehicle-logbook/:id/signature?type=pickup|return
router.get("/:id/signature", async (req, res) => {
  const logbook_id = req.params.id;
  const { type } = req.query;

  if (!["pickup", "return"].includes(type)) {
    return res.status(400).json({ error: "Invalid type." });
  }

  try {
    const [rows] = await pool.execute(
      `SELECT file_path FROM vehicle_logbook_signatures 
       WHERE logbook_id = ? AND type = ?
       ORDER BY id DESC LIMIT 1`,
      [logbook_id, type]
    );

    if (!rows.length) {
      return res.status(404).json({ error: "Υπογραφή δεν βρέθηκε." });
    }

    const url = "/" + rows[0].file_path.replace(/\\/g, "/");
    res.json({ url });
  } catch (err) {
    console.error("❌ Signature fetch error:", err);
    res.status(500).json({ error: "Σφάλμα ανάκτησης υπογραφής." });
  }
});

// 📌 GET /api/vehicle-logbook/:id/signatures
router.get("/:id/signatures", async (req, res) => {
  const logbook_id = req.params.id;

  try {
    const [rows] = await pool.execute(
      `SELECT type, file_path FROM vehicle_logbook_signatures WHERE logbook_id = ?`,
      [logbook_id]
    );
    res.json(rows);
  } catch (err) {
    console.error("❌ Signature fetch error:", err);
    res.status(500).json({ error: "Σφάλμα ανάκτησης υπογραφών." });
  }
});

export default router;
