import express from "express";
import db from "../db.js";
import {
  createCashSession,
  getCashSessions,
  getCashSessionById,
  updateCashSession,
  deleteCashSession,
} from "../controllers/cashController.js";
import { requireAuth, requireAuth as verifyToken } from "../middleware/authMiddleware.js";
import { uploadExpense } from "../utils/upload.js";
import { uploadReceiptHandler } from "../controllers/cashController.js";

const router = express.Router();

// === CRUD & uploads ===
router.post("/", requireAuth, uploadExpense.array("expense_files"), createCashSession);
router.put("/:id", requireAuth, uploadExpense.array("expense_files"), updateCashSession);
router.get("/", requireAuth, getCashSessions);
router.get("/:id", requireAuth, getCashSessionById);
router.delete("/:id", requireAuth, deleteCashSession);
router.post(
  "/:sessionId/receipt",
  requireAuth,
  uploadExpense.single("receipt_file"),
  uploadReceiptHandler
);

// === DASHBOARD STATS ===
router.get("/stats/dashboard", verifyToken, async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT 
        b.name AS branch_name,
        cs.cash_type,
        COUNT(cs.id) AS total_sessions,
        SUM(cs.system_cash) AS total_system,
        SUM(cs.declared_cash) AS total_declared,
        SUM(cs.declared_cash - cs.system_cash) AS total_diff,
        SUM(cs.pos_charges + cs.pos_cod + cs.pos_fastpay) AS total_pos,
        SUM(cs.cash_charges + cs.cash_cod + cs.cash_fastpay) AS total_cash
      FROM cash_sessions cs
      JOIN branches b ON b.id = cs.branch_id
      GROUP BY b.id, cs.cash_type
      ORDER BY b.name ASC;
    `);
    res.json(rows);
  } catch (err) {
    console.error("Error fetching cash dashboard stats:", err);
    res.status(500).json({ message: "Database error: " + err.message });
  }
});

// === DAILY REPORTS WITH GROUPING ===
router.get("/reports/daily", requireAuth, async (req, res) => {
  try {
    const { date } = req.query;
    if (!date) {
      return res.status(400).json({ message: "Missing date parameter (YYYY-MM-DD)" });
    }

    // 🧮 Κεντρικό query με join στα έξοδα
    const [rows] = await db.query(`
      SELECT 
        b.name AS branch_name,
        cs.cash_type,
        u.username AS user_name,
        cs.system_cash,
        cs.declared_cash,
        IFNULL(SUM(ce.amount), 0) AS total_expenses,
        ((cs.declared_cash + IFNULL(SUM(ce.amount), 0)) - cs.system_cash) AS diff
      FROM cash_sessions cs
      JOIN branches b ON b.id = cs.branch_id
      JOIN users u ON u.id = cs.user_id
      LEFT JOIN cash_expenses ce ON ce.session_id = cs.id
      WHERE DATE(cs.session_date) = ?
      GROUP BY cs.id, b.name, cs.cash_type, u.username, cs.system_cash, cs.declared_cash
      ORDER BY b.name, cs.cash_type;
    `, [date]);

    if (!rows.length) {
      return res.json({
        message: "No sessions found for that date",
        summary: { total_declared: 0, total_system: 0, total_diff: 0 },
        grouped: [],
      });
    }

    // ➕ Cast και ομαδοποίηση
    const data = rows.map(r => ({
      ...r,
      system_cash: Number(r.system_cash ?? 0),
      declared_cash: Number(r.declared_cash ?? 0),
      total_expenses: Number(r.total_expenses ?? 0),
      diff: Number(r.diff ?? 0),
    }));

    const grouped = {};
    data.forEach(r => {
      if (!grouped[r.branch_name]) {
        grouped[r.branch_name] = {
          branch_name: r.branch_name,
          types: {},
          subtotal_system: 0,
          subtotal_declared: 0,
          subtotal_expenses: 0,
          subtotal_diff: 0,
        };
      }

      const branch = grouped[r.branch_name];
      if (!branch.types[r.cash_type]) {
        branch.types[r.cash_type] = {
          cash_type: r.cash_type,
          total_system: 0,
          total_declared: 0,
          total_expenses: 0,
          total_diff: 0,
        };
      }

      const type = branch.types[r.cash_type];
      type.total_system += r.system_cash;
      type.total_declared += r.declared_cash;
      type.total_expenses += r.total_expenses;
      type.total_diff += r.diff;

      branch.subtotal_system += r.system_cash;
      branch.subtotal_declared += r.declared_cash;
      branch.subtotal_expenses += r.total_expenses;
      branch.subtotal_diff += r.diff;
    });

    const groupedArray = Object.values(grouped).map(b => ({
      ...b,
      types: Object.values(b.types),
    }));

    // ✅ Γενικά σύνολα
    const summary = groupedArray.reduce(
      (acc, b) => {
        acc.total_declared += b.subtotal_declared;
        acc.total_system += b.subtotal_system;
        acc.total_expenses += b.subtotal_expenses;
        acc.total_diff += b.subtotal_diff;
        return acc;
      },
      { total_declared: 0, total_system: 0, total_expenses: 0, total_diff: 0 }
    );

    res.json({ summary, grouped: groupedArray });
  } catch (err) {
    console.error("Error fetching daily cash report:", err);
    res.status(500).json({ message: "Database error: " + err.message });
  }
});



export default router;
