import pool from "../db.js";
import fs from "fs";
import path from "path";

// Δημιουργία παραλαβής οχήματος
export async function createLogbookEntry(req, res) {
  const { vehicle_id, user_id, branch_id, pickup_time, start_km, notes, has_incident } = req.body;
  const file = req.file;
  const signature_path = file ? `/uploads/signatures/${file.filename}` : null;

  const connection = await pool.getConnection();
  try {
    await connection.beginTransaction();

    // 1. Καταχώρηση παραλαβής
    const [result] = await connection.execute(
      `INSERT INTO vehicle_logbook (vehicle_id, user_id, branch_id, pickup_time, start_km, notes, has_incident)
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [vehicle_id, user_id, branch_id, pickup_time, start_km, notes, has_incident || false]
    );

    const logbookId = result.insertId;

    // 2. Αν υπάρχει υπογραφή, καταχώρησέ την
    if (signature_path) {
      await connection.execute(
        `INSERT INTO vehicle_logbook_signatures (logbook_id, type, signature_path)
         VALUES (?, 'pickup', ?)`,
        [logbookId, signature_path]
      );
    }

    await connection.commit();
    res.status(201).json({ message: "Καταχωρήθηκε η παραλαβή.", id: logbookId });
  } catch (error) {
    await connection.rollback();
    console.error("createLogbookEntry error:", error);
    res.status(500).json({ error: "Σφάλμα καταχώρησης παραλαβής." });
  } finally {
    connection.release();
  }
}

// Επιστροφή οχήματος
export async function returnVehicle(req, res) {
  const { id } = req.params;
  const { return_time, end_km, notes, has_incident } = req.body;
  const file = req.file;
  const signature_path = file ? `/uploads/signatures/${file.filename}` : null;

  const connection = await pool.getConnection();
  try {
    await connection.beginTransaction();

    // 1. Ενημέρωση επιστροφής
    await connection.execute(
      `UPDATE vehicle_logbook
       SET return_time = ?, end_km = ?, notes = ?, has_incident = ?
       WHERE id = ?`,
      [return_time, end_km, notes, has_incident || false, id]
    );

    // 2. Αν υπάρχει υπογραφή, καταχώρησέ την
    if (signature_path) {
      await connection.execute(
        `INSERT INTO vehicle_logbook_signatures (logbook_id, type, signature_path)
         VALUES (?, 'return', ?)`,
        [id, signature_path]
      );
    }

    await connection.commit();
    res.json({ message: "Η επιστροφή καταχωρήθηκε." });
  } catch (error) {
    await connection.rollback();
    console.error("returnVehicle error:", error);
    res.status(500).json({ error: "Σφάλμα καταχώρησης επιστροφής." });
  } finally {
    connection.release();
  }
}

// Προβολή εγγραφών logbook
export async function getLogbookEntries(req, res) {
  try {
    const [rows] = await pool.execute(`
      SELECT l.*, v.plate, u.username
      FROM vehicle_logbook l
      JOIN vehicles v ON l.vehicle_id = v.id
      JOIN users u ON l.user_id = u.id
      ORDER BY l.pickup_time DESC
    `);
    res.json(rows);
  } catch (error) {
    console.error("getLogbookEntries error:", error);
    res.status(500).json({ error: "Σφάλμα λήψης εγγραφών." });
  }
}

// Προβολή υπογραφής
export async function getLogbookSignature(req, res) {
  const { id } = req.params;
  const { type } = req.query;

  try {
    const [rows] = await pool.execute(
      `SELECT signature_path FROM vehicle_logbook_signatures
       WHERE logbook_id = ? AND type = ?`,
      [id, type]
    );

    if (rows.length === 0) {
      return res.status(404).json({ message: "Δεν βρέθηκε υπογραφή." });
    }

    res.json({ url: rows[0].signature_path });
  } catch (error) {
    console.error("getLogbookSignature error:", error);
    res.status(500).json({ error: "Σφάλμα ανάκτησης υπογραφής." });
  }
}

// Προβολή εγγραφής ανά ID
export async function getLogbookEntryById(req, res) {
  const { id } = req.params;

  try {
    const [rows] = await pool.execute(`SELECT * FROM vehicle_logbook WHERE id = ?`, [id]);
    if (rows.length === 0) {
      return res.status(404).json({ error: "Δεν βρέθηκε η εγγραφή." });
    }
    res.json(rows[0]);
  } catch (error) {
    console.error("getLogbookEntryById error:", error);
    res.status(500).json({ error: "Σφάλμα." });
  }
}

// Διαγραφή logbook εγγραφής και υπογραφών
export async function deleteLogbookEntry(req, res) {
  const { id } = req.params;

  try {
    await pool.execute("DELETE FROM vehicle_logbook_signatures WHERE logbook_id = ?", [id]);
    await pool.execute("DELETE FROM vehicle_logbook WHERE id = ?", [id]);

    res.status(200).json({ message: "Η εγγραφή διαγράφηκε." });
  } catch (err) {
    console.error("❌ Σφάλμα διαγραφής logbook:", err);
    res.status(500).json({ message: "Σφάλμα διαγραφής.", error: err });
  }
}

// Ενημέρωση εγγραφής logbook
export async function updateLogbookEntry(req, res) {
  const { id } = req.params;
  const { pickup_time, return_time, start_km, end_km, notes, has_incident } = req.body;

  try {
    await pool.execute(
      `UPDATE vehicle_logbook
       SET pickup_time = ?, return_time = ?, start_km = ?, end_km = ?, notes = ?, has_incident = ?
       WHERE id = ?`,
      [pickup_time, return_time, start_km, end_km, notes, has_incident, id]
    );
    res.json({ message: "Η εγγραφή ενημερώθηκε." });
  } catch (error) {
    console.error("updateLogbookEntry error:", error);
    res.status(500).json({ error: "Σφάλμα κατά την ενημέρωση." });
  }
}
