// src/controllers/vehicleDocumentController.js
import pool from "../db.js";
import fs from "fs";
import path from "path";

// 📥 Upload
export async function uploadDocument(req, res) {
  try {
    const { vehicleId } = req.params;
    const file = req.file;
    if (!file) return res.status(400).json({ message: "No file uploaded" });

    const conn = await pool.getConnection();
    await conn.query(
      "INSERT INTO vehicle_documents (vehicle_id, filename, filepath) VALUES (?, ?, ?)",
      [vehicleId, file.originalname, file.path]
    );
    conn.release();
    return res.status(201).json({ message: "File uploaded" });
  } catch (err) {
    console.error("❌ uploadDocument:", err);
    return res.status(500).json({ message: "Server error" });
  }
}

// 📂 List
export async function getVehicleDocuments(req, res) {
  try {
    const { vehicleId } = req.params;
    const conn = await pool.getConnection();
    const [rows] = await conn.query(
      "SELECT id, vehicle_id, filename, filepath, created_at FROM vehicle_documents WHERE vehicle_id = ? ORDER BY id DESC",
      [vehicleId]
    );
    conn.release();
    return res.json(rows);
  } catch (err) {
    console.error("❌ getVehicleDocuments:", err);
    return res.status(500).json({ message: "Server error" });
  }
}

// ⬇ Download by docId
export async function downloadDocumentById(req, res) {
  try {
    const { vehicleId, docId } = req.params;
    const conn = await pool.getConnection();
    const [[doc]] = await conn.query(
      "SELECT id, vehicle_id, filename, filepath FROM vehicle_documents WHERE id = ?",
      [docId]
    );
    conn.release();

    if (!doc) return res.status(404).json({ message: "Document not found" });
    if (String(doc.vehicle_id) !== String(vehicleId))
      return res.status(403).json({ message: "Vehicle mismatch" });

    if (!fs.existsSync(doc.filepath))
      return res.status(410).json({ message: "File missing on disk" });

    return res.download(doc.filepath, doc.filename);
  } catch (err) {
    console.error("❌ downloadDocumentById:", err);
    return res.status(500).json({ message: "Server error" });
  }
}

// 🗑️ Delete by docId
export async function deleteDocumentById(req, res) {
  try {
    const { vehicleId, docId } = req.params;
    const conn = await pool.getConnection();
    const [[doc]] = await conn.query(
      "SELECT id, vehicle_id, filepath FROM vehicle_documents WHERE id = ?",
      [docId]
    );
    if (!doc) {
      conn.release();
      return res.status(404).json({ message: "Document not found" });
    }
    if (String(doc.vehicle_id) !== String(vehicleId)) {
      conn.release();
      return res.status(403).json({ message: "Vehicle mismatch" });
    }

    if (doc.filepath && fs.existsSync(doc.filepath)) {
      try { fs.unlinkSync(doc.filepath); } catch (_) { /* ignore */ }
    }
    await conn.query("DELETE FROM vehicle_documents WHERE id = ?", [docId]);
    conn.release();

    return res.json({ message: "Document deleted" });
  } catch (err) {
    console.error("❌ deleteDocumentById:", err);
    return res.status(500).json({ message: "Server error" });
  }
}
