import bcrypt from "bcryptjs";
import pool from "../db.js";

// Get all users (admin sees all, branch_admin sees their branch)
export async function getUsers(req, res) {
  try {
    const conn = await pool.getConnection();
    let users;
    if (req.user.role === "admin") {
      [users] = await conn.query("SELECT id, username, role, branch_id FROM users");
    } else if (req.user.role === "branch_admin") {
      [users] = await conn.query(
        "SELECT id, username, role, branch_id FROM users WHERE branch_id = ?",
        [req.user.branch_id]
      );
    } else {
      return res.status(403).json({ message: "Forbidden" });
    }
    res.json(users);
  } catch (err) {
    console.error("❌ Error getting users:", err);
    res.status(500).json({ message: "Server error" });
  }
}

// Create new user
export async function createUser(req, res) {
  const { username, password, role, branch_id } = req.body;
  const finalBranchId = branch_id === "" ? null : branch_id;

  try {
    const hashed = await bcrypt.hash(password, 10);
    const conn = await pool.getConnection();
    await conn.query(
      "INSERT INTO users (username, password, role, branch_id) VALUES (?, ?, ?, ?)",
      [username, hashed, role, finalBranchId]
    );
    res.status(201).json({ message: "User created" });
  } catch (err) {
    console.error("❌ Error creating user:", err);
    res.status(500).json({ message: "Server error" });
  }
}


// Update user info (με διατήρηση του παλιού password αν δεν δοθεί νέο)
export async function updateUser(req, res) {
  const { id } = req.params;
  const { username, password, role, branch_id } = req.body;
  const finalBranchId = branch_id === "" ? null : branch_id;

  try {
    const conn = await pool.getConnection();

    if (password && password.trim() !== "") {
      const hashed = await bcrypt.hash(password, 10);
      await conn.query(
        "UPDATE users SET username = ?, password = ?, role = ?, branch_id = ? WHERE id = ?",
        [username, hashed, role, finalBranchId, id]
      );
    } else {
      await conn.query(
        "UPDATE users SET username = ?, role = ?, branch_id = ? WHERE id = ?",
        [username, role, finalBranchId, id]
      );
    }

    res.json({ message: "User updated" });
  } catch (err) {
    console.error("❌ Error updating user:", err);
    res.status(500).json({ message: "Server error" });
  }
}


// Change password (by user)
export async function changePassword(req, res) {
  const { oldPassword, newPassword } = req.body;
  try {
    const conn = await pool.getConnection();
    const [rows] = await conn.query("SELECT * FROM users WHERE id = ?", [req.user.id]);
    const user = rows[0];
    const match = await bcrypt.compare(oldPassword, user.password);
    if (!match) return res.status(401).json({ message: "Wrong old password" });

    const hashed = await bcrypt.hash(newPassword, 10);
    await conn.query("UPDATE users SET password = ? WHERE id = ?", [hashed, req.user.id]);
    res.json({ message: "Password updated" });
  } catch (err) {
    console.error("❌ Error changing password:", err);
    res.status(500).json({ message: "Server error" });
  }
}
export async function deleteUser(req, res) {
  const { id } = req.params;
  try {
    const conn = await pool.getConnection();
    await conn.query("DELETE FROM users WHERE id = ?", [id]);
    res.json({ message: "User deleted" });
  } catch (err) {
    console.error("❌ Error deleting user:", err);
    res.status(500).json({ message: "Server error" });
  }
}
export async function getAllUsers(req, res) {
  try {
    const conn = await pool.getConnection();
    const [rows] = await conn.query("SELECT id, username, role, branch_id, created_at FROM users");
    res.json(rows);
  } catch (err) {
    console.error("❌ Error fetching users:", err);
    res.status(500).json({ message: "Server error" });
  }
}