import db from "../db.js";

// === Λήψη όλων των πληρωμών ενός item ===
export const getPaymentsByItem = async (req, res) => {
  const { itemId } = req.params;
  try {
    const [rows] = await db.query(
      `
      SELECT 
        p.id,
        p.payment_date,
        p.amount,
        p.method,
        p.notes
      FROM employee_payroll_payments p
      WHERE p.item_id = ?
      ORDER BY p.payment_date DESC
      `,
      [itemId]
    );
    res.json(rows);
  } catch (err) {
    console.error("❌ getPaymentsByItem error:", err);
    res.status(500).json({ error: "Database error" });
  }
};

// === Δημιουργία νέας πληρωμής ===
export const createPayment = async (req, res) => {
  try {
    const { payroll_id, item_id, amount, payment_date, method, notes } = req.body;

    if (!payroll_id || !amount || amount <= 0) {
      return res
        .status(400)
        .json({ error: "Λείπουν υποχρεωτικά πεδία ή μη έγκυρο ποσό" });
    }

    // 🔹 Βρες τον υπάλληλο από το payroll
    const [[payroll]] = await db.query(
      "SELECT employee_id FROM employee_payrolls WHERE id = ?",
      [payroll_id]
    );
    if (!payroll) return res.status(404).json({ error: "Payroll not found" });

    const employee_id = payroll.employee_id;

    // 🔹 Εισαγωγή πληρωμής
    await db.query(
      `
      INSERT INTO employee_payroll_payments 
      (employee_id, payroll_id, item_id, amount, payment_date, method, notes)
      VALUES (?, ?, ?, ?, ?, ?, ?)
      `,
      [
        employee_id,
        payroll_id,
        item_id || null,
        amount,
        payment_date || new Date(),
        method || "bank",
        notes || "",
      ]
    );

    // 🔹 Ενημέρωση του amount_paid στο item (remaining υπολογίζεται αυτόματα στη DB)
    if (item_id) {
      await db.query(
        `
        UPDATE employee_payroll_items
        SET amount_paid = (
          SELECT COALESCE(SUM(amount), 0)
          FROM employee_payroll_payments
          WHERE item_id = ?
        )
        WHERE id = ?
        `,
        [item_id, item_id]
      );
    }

    res.json({ message: "✅ Payment added successfully" });
  } catch (err) {
    console.error("❌ createPayment error:", err);
    res.status(500).json({ error: err.message });
  }
};

// === Διαγραφή πληρωμής ===
export const deletePayment = async (req, res) => {
  const { id } = req.params;
  try {
    const [[payment]] = await db.query(
      "SELECT item_id FROM employee_payroll_payments WHERE id = ?",
      [id]
    );

    await db.query("DELETE FROM employee_payroll_payments WHERE id = ?", [id]);

    // 🔹 Ενημέρωση amount_paid στο item (remaining ενημερώνεται αυτόματα)
    if (payment?.item_id) {
      await db.query(
        `
        UPDATE employee_payroll_items
        SET amount_paid = (
          SELECT COALESCE(SUM(amount), 0)
          FROM employee_payroll_payments
          WHERE item_id = ?
        )
        WHERE id = ?
        `,
        [payment.item_id, payment.item_id]
      );
    }

    res.json({ message: "🗑️ Payment deleted and recalculated" });
  } catch (err) {
    console.error("❌ deletePayment error:", err);
    res.status(500).json({ error: "Database error" });
  }
};
