import db from "../db.js";

// === Καρτέλα Πελάτη με σωρευτικό υπόλοιπο και totals ===
export async function getCustomerLedger(req, res) {
  try {
    const { id } = req.params;
    const { from, to } = req.query;

    // Υπόλοιπο έναρξης (μόνο αν υπάρχει 'from')
    let openingBalance = 0;
    if (from && from.trim() !== "") {
      const [beforeRows] = await db.query(
        `
        SELECT COALESCE(SUM(debit - credit), 0) AS opening_balance
        FROM customer_ledger
        WHERE customer_id = ? AND txn_date < ?
        `,
        [id, from]
      );
      openingBalance = Number(beforeRows[0].opening_balance) || 0;
    }

    // Δημιουργία ασφαλούς WHERE
    let whereClause = "WHERE customer_id = ?";
    const params = [id];

    if (from && from.trim() !== "") {
      whereClause += " AND txn_date >= ?";
      params.push(from);
    }
    if (to && to.trim() !== "") {
      whereClause += " AND txn_date <= ?";
      params.push(to);
    }

    const [rows] = await db.query(
      `
      SELECT id, txn_date, reference_number, description, debit, credit
      FROM customer_ledger
      ${whereClause}
      ORDER BY txn_date ASC, id ASC
      `,
      params
    );

    // Υπολογισμός σωρευτικού υπολοίπου & συνόλων
    let running = openingBalance;
    let totalDebit = 0;
    let totalCredit = 0;

    const entries = rows.map((r) => {
      const debit = Number(r.debit || 0);
      const credit = Number(r.credit || 0);
      totalDebit += debit;
      totalCredit += credit;
      running += debit - credit;
      return { ...r, balance: running };
    });

    res.json({
      opening_balance: openingBalance,
      closing_balance: running,
      total_debit: totalDebit,
      total_credit: totalCredit,
      entries,
    });
  } catch (err) {
    console.error("❌ Error fetching ledger:", err);
    res.status(500).json({ message: err.message });
  }
}

// === Αναζήτηση καρτέλας πελάτη (με φίλτρα & totals) ===
export async function searchLedger(req, res) {
  try {
    let { q, from, to } = req.query;

    if (!q) {
      return res
        .status(400)
        .json({ message: "Απαιτείται αναζήτηση (όνομα, ΑΦΜ ή κωδικός)" });
    }

    // ✅ Καθαρισμός q (αφαιρούμε παρενθέσεις π.χ. “(123)”)
    q = q.replace(/\(.*?\)/g, "").trim();

    // 🔹 Εύρεση πελάτη
    const [customers] = await db.query(
      `
      SELECT id, name, code, vat_number 
      FROM customers
      WHERE name LIKE ? OR code LIKE ? OR vat_number LIKE ?
      LIMIT 1
      `,
      [`%${q}%`, `%${q}%`, `%${q}%`]
    );

    if (customers.length === 0) {
      return res.json({ entries: [], balance: 0 });
    }

    const customerId = customers[0].id;

    // 🔹 Υπόλοιπο έναρξης — Υπολογίζεται ΜΟΝΟ αν υπάρχει 'from'
    let openingBalance = 0;
    if (from && from.trim() !== "") {
      const [beforeRows] = await db.query(
        `
        SELECT COALESCE(SUM(debit - credit), 0) AS opening_balance
        FROM customer_ledger
        WHERE customer_id = ? AND txn_date < ?
        `,
        [customerId, from]
      );
      openingBalance = Number(beforeRows[0].opening_balance) || 0;
    }

    // 🔹 Κατασκευή ασφαλούς query
    let whereClause = "WHERE customer_id = ?";
    const params = [customerId];

    if (from && from.trim() !== "") {
      whereClause += " AND txn_date >= ?";
      params.push(from);
    }
    if (to && to.trim() !== "") {
      whereClause += " AND txn_date <= ?";
      params.push(to);
    }

    const [rows] = await db.query(
      `
      SELECT txn_date, reference_number, description, debit, credit
      FROM customer_ledger
      ${whereClause}
      ORDER BY txn_date ASC
      `,
      params
    );

    // 🔹 Υπολογισμός υπολοίπου & συνόλων
    let runningBalance = openingBalance;
    let totalDebit = 0;
    let totalCredit = 0;

    const entries = rows.map((r) => {
      const debit = Number(r.debit) || 0;
      const credit = Number(r.credit) || 0;
      totalDebit += debit;
      totalCredit += credit;
      runningBalance += debit - credit;
      return { ...r, debit, credit, balance: runningBalance };
    });

    res.json({
      customer: customers[0],
      opening_balance: openingBalance,
      total_debit: totalDebit,
      total_credit: totalCredit,
      period_balance: totalDebit - totalCredit,
      closing_balance: runningBalance,
      entries,
    });
  } catch (err) {
    console.error("❌ Error searching ledger:", err);
    res.status(500).json({ message: err.message });
  }
}
