import db from "../db.js";
import { isUserApprover } from "../utils/leaveHelpers.js";

export const createLeaveRequest = async (req, res) => {
  const {
    user_id,
    branch_id,
    from_date,
    to_date,
    type,
    reason,
  } = req.body;

  try {
    const [result] = await db.execute(
      `INSERT INTO leave_requests 
        (user_id, branch_id, from_date, to_date, type, reason) 
        VALUES (?, ?, ?, ?, ?, ?)`,
      [user_id, branch_id, from_date, to_date, type, reason]
    );

    res.status(201).json({ id: result.insertId, message: "Αίτημα άδειας καταχωρήθηκε." });
  } catch (error) {
    console.error("createLeaveRequest error:", error);
    res.status(500).json({ message: "Σφάλμα κατά την καταχώριση." });
  }
};

export const getLeaveRequests = async (req, res) => {
  try {
    const [rows] = await db.execute(`
      SELECT lr.*, u.username, b.name as branch_name
      FROM leave_requests lr
      LEFT JOIN users u ON lr.user_id = u.id
      LEFT JOIN branches b ON lr.branch_id = b.id
      ORDER BY lr.created_at DESC
    `);
    res.json(rows);
  } catch (error) {
    console.error("getLeaveRequests error:", error);
    res.status(500).json({ message: "Σφάλμα κατά την ανάκτηση." });
  }
};

export const getMyLeaves = async (req, res) => {
  const userId = req.user.id;

  try {
    const [rows] = await db.execute(
      `SELECT * FROM leave_requests WHERE user_id = ? ORDER BY created_at DESC`,
      [userId]
    );
    res.json(rows);
  } catch (error) {
    console.error("getMyLeaves error:", error);
    res.status(500).json({ message: "Σφάλμα κατά την ανάκτηση." });
  }
};

export const approveLeave = async (req, res) => {
  const { id } = req.params;
  const { level } = req.query;
  const user = req.user;

  if (!["1", "2", "3"].includes(level)) {
    return res.status(400).json({ message: "Μη έγκυρο επίπεδο έγκρισης." });
  }

  try {
    const [rows] = await db.execute(`SELECT * FROM leave_requests WHERE id = ?`, [id]);
    if (rows.length === 0) return res.status(404).json({ message: "Δεν βρέθηκε η άδεια." });

    const leave = rows[0];

    // Αν δεν είναι admin, έλεγξε αν είναι εγκριτής για αυτό το level & υποκατάστημα
    if (user.role !== "admin") {
      const authorized = await isUserApprover(user.id, leave.branch_id, level);
      if (!authorized) {
        return res.status(403).json({ message: "Δεν έχετε δικαίωμα έγκρισης για αυτό το επίπεδο." });
      }
    }

    const column = `status_level${level}`;
    await db.execute(`UPDATE leave_requests SET ${column} = 'approved' WHERE id = ?`, [id]);

    res.json({ message: "Η άδεια εγκρίθηκε." });
  } catch (error) {
    console.error("approveLeave error:", error);
    res.status(500).json({ message: "Σφάλμα κατά την έγκριση." });
  }
};

export const rejectLeave = async (req, res) => {
  const { id } = req.params;
  const { level } = req.query;
  const user = req.user;

  if (!["1", "2", "3"].includes(level)) {
    return res.status(400).json({ message: "Μη έγκυρο επίπεδο απόρριψης." });
  }

  try {
    const [rows] = await db.execute(`SELECT * FROM leave_requests WHERE id = ?`, [id]);
    if (rows.length === 0) return res.status(404).json({ message: "Δεν βρέθηκε η άδεια." });

    const leave = rows[0];

    if (user.role !== "admin") {
      const authorized = await isUserApprover(user.id, leave.branch_id, level);
      if (!authorized) {
        return res.status(403).json({ message: "Δεν έχετε δικαίωμα απόρριψης για αυτό το επίπεδο." });
      }
    }

    const column = `status_level${level}`;
    await db.execute(`UPDATE leave_requests SET ${column} = 'rejected' WHERE id = ?`, [id]);

    res.json({ message: "Η άδεια απορρίφθηκε." });
  } catch (error) {
    console.error("rejectLeave error:", error);
    res.status(500).json({ message: "Σφάλμα κατά την απόρριψη." });
  }
};
