import db from "../db.js";

// === Λήψη όλων των τιμολογίων ===
export async function getInvoices(req, res) {
  try {
    const [rows] = await db.query(`
      SELECT 
        i.*, 
        c.name AS customer_name,
        FORMAT(i.amount, 2) AS formatted_amount,
        CASE 
  WHEN i.is_paid = 1 THEN 'Εξοφλημένο'
  WHEN (
    SELECT COALESCE(SUM(p.amount),0) 
    FROM invoice_payments p 
    WHERE p.customer_id = i.customer_id
  ) >= i.amount THEN 'Εξοφλημένο'
  WHEN EXISTS (
    SELECT 1 FROM invoice_payments p 
    WHERE p.customer_id = i.customer_id
  ) THEN 'Μερικώς'
  ELSE 'Ανεξόφλητο'
END AS status

      FROM invoices i
      JOIN customers c ON c.id = i.customer_id
      ORDER BY i.issue_date DESC
    `);
    res.json(rows);
  } catch (err) {
    console.error("❌ Error fetching invoices:", err);
    res.status(500).json({ message: err.message });
  }
}

// === Δημιουργία νέου τιμολογίου ===
export async function createInvoice(req, res) {
  const conn = await db.getConnection();
  await conn.beginTransaction();

  try {
    const {
      customer_id,
      invoice_number,
      type,
      elta_code,
      issue_date,
      amount,
      notes,
    } = req.body;
// === Έλεγχος για διπλότυπο τιμολόγιο ===
const [exists] = await conn.query(
  `SELECT id FROM invoices 
   WHERE customer_id = ? AND invoice_number = ? 
   LIMIT 1`,
  [customer_id, invoice_number]
);

if (exists.length > 0) {
  await conn.rollback();
  return res
    .status(400)
    .json({ message: "Υπάρχει ήδη τιμολόγιο με αυτόν τον αριθμό για τον συγκεκριμένο πελάτη." });
}

    // ✅ Κανονικοποίηση τύπου τιμολογίου
    const normalizedType =
      type === "ΠΙΣΤΩΤΙΚΟ" || type === "credit_invoice"
        ? "credit_invoice"
        : "service_invoice";

    // ✅ Κανονικοποίηση τύπου για το ledger
    const ledgerType =
      normalizedType === "credit_invoice" ? "credit_invoice" : "invoice";

    // 1️⃣ Καταχώρηση τιμολογίου
    const [result] = await conn.query(
      `INSERT INTO invoices 
       (customer_id, invoice_number, type, elta_code, issue_date, amount, notes)
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [
        customer_id,
        invoice_number,
        normalizedType,
        elta_code,
        issue_date,
        amount,
        notes,
      ]
    );

    const invoice_id = result.insertId;

    // 2️⃣ Καταχώρηση στο customer_ledger
    await conn.query(
      `INSERT INTO customer_ledger 
       (customer_id, reference_type, reference_id, reference_number, description, txn_date, debit, credit)
       VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        customer_id,
        ledgerType,
        invoice_id,
        invoice_number,
        normalizedType === "credit_invoice"
          ? `Πιστωτικό Τιμολόγιο #${invoice_number}`
          : `ΤΠΥ #${invoice_number}`,
        issue_date,
        normalizedType === "credit_invoice" ? 0 : amount,
        normalizedType === "credit_invoice" ? amount : 0,
      ]
    );

    await conn.commit();
    res.status(201).json({ message: "✅ Το τιμολόγιο καταχωρήθηκε επιτυχώς" });
  } catch (err) {
    await conn.rollback();
    console.error("❌ Error creating invoice:", err);
    res.status(500).json({ message: err.message });
  } finally {
    conn.release();
  }
}

// === Διαγραφή τιμολογίου ===
export async function deleteInvoice(req, res) {
  const conn = await db.getConnection();
  await conn.beginTransaction();

  try {
    const { id } = req.params;

    // 1️⃣ Έλεγχος ύπαρξης τιμολογίου
    const [[invoice]] = await conn.query(
      "SELECT customer_id, invoice_number, type FROM invoices WHERE id = ?",
      [id]
    );

    if (!invoice) {
      await conn.rollback();
      return res.status(404).json({ message: "❌ Το τιμολόγιο δεν βρέθηκε." });
    }

    // 2️⃣ Διαγραφή αντίστοιχης εγγραφής από το ledger
    const ledgerType =
      invoice.type === "credit_invoice" ? "credit_invoice" : "invoice";

    await conn.query(
      `DELETE FROM customer_ledger 
       WHERE reference_type = ? AND reference_id = ?`,
      [ledgerType, id]
    );

    // 3️⃣ Διαγραφή του ίδιου του τιμολογίου
    await conn.query("DELETE FROM invoices WHERE id = ?", [id]);

    await conn.commit();
    res.json({
      message: "🗑️ Το τιμολόγιο και η εγγραφή στο λογιστικό βιβλίο διαγράφηκαν.",
    });
  } catch (err) {
    await conn.rollback();
    console.error("❌ Error deleting invoice:", err);
    res.status(500).json({ message: err.message });
  } finally {
    conn.release();
  }
}

// === Προβολή συγκεκριμένου τιμολογίου ===
export async function getInvoiceById(req, res) {
  try {
    const { id } = req.params;

    const [[invoice]] = await db.query(
      `SELECT i.*, c.name AS customer_name
       FROM invoices i
       JOIN customers c ON c.id = i.customer_id
       WHERE i.id = ?`,
      [id]
    );

    if (!invoice)
      return res.status(404).json({ message: "Δεν βρέθηκε το τιμολόγιο" });

    const [payments] = await db.query(
      `SELECT * FROM invoice_payments WHERE invoice_id = ? ORDER BY payment_date ASC`,
      [id]
    );

    res.json({ invoice, payments });
  } catch (err) {
    console.error("❌ Error fetching invoice:", err);
    res.status(500).json({ message: err.message });
  }
}
