import db from "../db.js";

/**
 * 🔹 GET /api/invoice-payments
 * Επιστρέφει όλες τις πληρωμές με στοιχεία πελάτη (και αν υπάρχει, τιμολόγιο)
 */
export async function getAllInvoicePayments(req, res) {
  try {
    const [rows] = await db.query(`
      SELECT 
        ip.id,
        ip.customer_id,
        c.name AS customer_name,
        ip.payment_date,
        ip.amount,
        ip.method,
        ip.notes,
        ip.created_at
      FROM invoice_payments ip
      LEFT JOIN customers c ON ip.customer_id = c.id
      ORDER BY ip.payment_date DESC, ip.id DESC
    `);
    res.json(rows);
  } catch (err) {
    console.error("❌ Error fetching payments:", err);
    res.status(500).json({ message: err.message });
  }
}

/**
 * 🔹 GET /api/invoice-payments/:id
 * Επιστρέφει συγκεκριμένη πληρωμή
 */
export async function getInvoicePayment(req, res) {
  try {
    const { id } = req.params;
    const [[payment]] = await db.query(
      `
      SELECT 
        ip.id,
        ip.customer_id,
        c.name AS customer_name,
        ip.payment_date,
        ip.amount,
        ip.method,
        ip.notes,
        ip.created_at
      FROM invoice_payments ip
      LEFT JOIN customers c ON ip.customer_id = c.id
      WHERE ip.id = ?
      `,
      [id]
    );

    if (!payment) return res.status(404).json({ message: "Η πληρωμή δεν βρέθηκε." });
    res.json(payment);
  } catch (err) {
    console.error("❌ Error fetching payment:", err);
    res.status(500).json({ message: err.message });
  }
}

/**
 * 🔹 POST /api/invoice-payments
 * Προσθέτει νέα πληρωμή πελάτη (σύνδεση ledger)
 */
export async function addInvoicePayment(req, res) {
  const conn = await db.getConnection();
  await conn.beginTransaction();

  try {
    const { customer_id, payment_date, amount, method, notes } = req.body;

    if (!customer_id || !payment_date || !amount)
      throw new Error("Λείπουν υποχρεωτικά πεδία.");

    // ✅ Καταχώρηση πληρωμής
    const [result] = await conn.query(
      `
      INSERT INTO invoice_payments (customer_id, payment_date, amount, method, notes)
      VALUES (?, ?, ?, ?, ?)
      `,
      [customer_id, payment_date, amount, method || "cash", notes || null]
    );

    const paymentId = result.insertId;

    // ✅ Καταχώρηση στην καρτέλα πελάτη
    const [[customer]] = await conn.query(
      "SELECT name FROM customers WHERE id = ?",
      [customer_id]
    );

    const descriptionMap = {
      cash: "Πληρωμή με Μετρητά",
      bank: "Κατάθεση μέσω Τράπεζας",
      card: "Πληρωμή με Κάρτα",
      elta_deposit: "Κατάθεση μέσω ΕΛΤΑ",
      elta_withholding: "Παρακράτηση ΕΛΤΑ",
      cod_withholding: "Παρακράτηση Αντικαταβολών",
    };

    const description = descriptionMap[method] || "Πληρωμή Πελάτη";

    await conn.query(
      `
      INSERT INTO customer_ledger
      (customer_id, reference_type, reference_id, description, txn_date, debit, credit)
      VALUES (?, 'payment', ?, ?, ?, 0, ?)
      `,
      [customer_id, paymentId, description, payment_date, amount]
    );

    await conn.commit();
    res.status(201).json({ message: "✅ Η πληρωμή καταχωρήθηκε επιτυχώς." });
  } catch (err) {
    await conn.rollback();
    console.error("❌ Error adding payment:", err);
    res.status(500).json({ message: err.message });
  } finally {
    conn.release();
  }
}

/**
 * 🔹 DELETE /api/invoice-payments/:id
 * Διαγράφει πληρωμή — το ledger καθαρίζεται αυτόματα με ON DELETE CASCADE
 */
export async function deleteInvoicePayment(req, res) {
  try {
    const { id } = req.params;

    const [result] = await db.query("DELETE FROM invoice_payments WHERE id = ?", [id]);

    if (result.affectedRows === 0)
      return res.status(404).json({ message: "Η πληρωμή δεν βρέθηκε." });

    // Ledger entry διαγράφεται αυτόματα από το ON DELETE CASCADE
    res.json({
      message: "🗑️ Η πληρωμή και η εγγραφή στο ledger διαγράφηκαν αυτόματα.",
    });
  } catch (err) {
    console.error("❌ Error deleting payment:", err);
    res.status(500).json({ message: err.message });
  }
}
export async function getPaymentsByCustomer(req, res) {
  try {
    const { id } = req.params;
    const [rows] = await db.query(
      `
      SELECT 
        ip.id,
        ip.customer_id,
        c.name AS customer_name,
        ip.payment_date,
        ip.amount,
        ip.method,
        ip.notes,
        ip.created_at
      FROM invoice_payments ip
      LEFT JOIN customers c ON ip.customer_id = c.id
      WHERE ip.customer_id = ?
      ORDER BY ip.payment_date DESC, ip.id DESC
      `,
      [id]
    );

    res.json(rows);
  } catch (err) {
    console.error("❌ Error fetching customer payments:", err);
    res.status(500).json({ message: err.message });
  }
}
