const db = require('../db');
const excel = require('exceljs');
const QRCode = require('qrcode');
const path = require('path');
const fs = require('fs');

exports.getAllEquipment = async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT e.*, em.brand, em.model, es.label AS status_label
      FROM equipment e
      LEFT JOIN equipment_models em ON e.model_id = em.id
      LEFT JOIN equipment_status es ON e.status_code = es.code
    `);
    res.json(rows);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

exports.getEquipmentById = async (req, res) => {
  try {
    const [rows] = await db.query('SELECT * FROM equipment WHERE id = ?', [req.params.id]);
    if (!rows.length) return res.status(404).json({ message: 'Equipment not found' });
    res.json(rows[0]);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

exports.createEquipment = async (req, res) => {
  try {
    const { system_serial, asset_tag, category_code, model_id, status_code, location_hint, notes } = req.body;
    await db.query(`
      INSERT INTO equipment (id, system_serial, asset_tag, category_code, model_id, status_code, location_hint, notes)
      VALUES (UUID(), ?, ?, ?, ?, ?, ?, ?)`,
      [system_serial, asset_tag, category_code, model_id, status_code, location_hint, notes]
    );
    res.json({ message: 'Equipment added successfully' });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

exports.updateEquipment = async (req, res) => {
  try {
    const { status_code, location_hint, notes } = req.body;
    await db.query(`
      UPDATE equipment SET status_code=?, location_hint=?, notes=?, updated_at=NOW()
      WHERE id=?`, [status_code, location_hint, notes, req.params.id]);
    res.json({ message: 'Equipment updated' });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

exports.deleteEquipment = async (req, res) => {
  try {
    await db.query(`DELETE FROM equipment WHERE id=?`, [req.params.id]);
    res.json({ message: 'Equipment deleted' });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

exports.assignEquipment = async (req, res) => {
  try {
    const { holder_id, holder_type, reason } = req.body;
    await db.query(`
      INSERT INTO equipment_assignments (id, equipment_id, holder_id, holder_type, reason)
      VALUES (UUID(), ?, ?, ?, ?)`, [req.params.id, holder_id, holder_type, reason]
    );
    res.json({ message: 'Equipment assigned successfully' });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

exports.unassignEquipment = async (req, res) => {
  try {
    await db.query(`
      UPDATE equipment_assignments 
      SET end_at = NOW() 
      WHERE equipment_id = ? AND end_at IS NULL`, [req.params.id]);
    res.json({ message: 'Equipment unassigned' });
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 📊 EXTRA FEATURE 1: Export to Excel
exports.exportEquipmentToExcel = async (req, res) => {
  try {
    const [rows] = await db.query(`SELECT * FROM equipment`);
    const workbook = new excel.Workbook();
    const worksheet = workbook.addWorksheet('Equipment');
    worksheet.columns = Object.keys(rows[0]).map(key => ({ header: key, key }));
    rows.forEach(r => worksheet.addRow(r));

    const filePath = path.join(__dirname, '../uploads/equipment_export.xlsx');
    await workbook.xlsx.writeFile(filePath);
    res.download(filePath);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 📈 EXTRA FEATURE 2: Dashboard Statistics
exports.getEquipmentStats = async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT status_code, COUNT(*) AS total 
      FROM equipment 
      GROUP BY status_code
    `);
    res.json(rows);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};

// 🔳 EXTRA FEATURE 3: QR Code Generator
exports.generateQRCode = async (req, res) => {
  try {
    const equipmentId = req.params.id;
    const qrPath = path.join(__dirname, `../uploads/qrcodes/${equipmentId}.png`);
    fs.mkdirSync(path.dirname(qrPath), { recursive: true });
    await QRCode.toFile(qrPath, `equipment:${equipmentId}`);
    res.sendFile(qrPath);
  } catch (err) {
    res.status(500).json({ message: err.message });
  }
};
