// controllers/customersController.js
import db from "../db.js";
import multer from "multer";
import XLSX from "xlsx";
import fs from "fs";

// === Multer setup ===
const upload = multer({ dest: "uploads/" });
export const uploadMiddleware = upload.single("file");

// === Λήψη όλων των πελατών ===
// === Λήψη πελατών με αναζήτηση & σελιδοποίηση ===
export async function getCustomers(req, res) {
  try {
    const { search = "", page = 1, limit = 20 } = req.query;
    const offset = (page - 1) * limit;

    let sql = "SELECT * FROM customers";
    const params = [];

    if (search) {
      sql += " WHERE name LIKE ? OR code LIKE ? OR vat_number LIKE ?";
      const searchTerm = `%${search}%`;
      params.push(searchTerm, searchTerm, searchTerm);
    }

    sql += " ORDER BY name ASC LIMIT ? OFFSET ?";
    params.push(Number(limit), Number(offset));

    const [rows] = await db.query(sql, params);

    // Για pagination πληροφορίες (πόσες σελίδες έχει)
    let countQuery = "SELECT COUNT(*) AS total FROM customers";
    if (search) countQuery += " WHERE name LIKE ? OR code LIKE ? OR vat_number LIKE ?";
    const [countRows] = await db.query(countQuery, search ? [ `%${search}%`, `%${search}%`, `%${search}%` ] : []);

    res.json({
      data: rows,
      total: countRows[0].total,
      currentPage: Number(page),
      totalPages: Math.ceil(countRows[0].total / limit),
    });
  } catch (err) {
    console.error("❌ Error fetching customers:", err);
    res.status(500).json({ message: "Σφάλμα κατά τη λήψη πελατών." });
  }
}


// === Δημιουργία πελάτη ===
export async function createCustomer(req, res) {
  try {
    const data = req.body;
    const {
      code,
      name,
      trade_name,
      profession,
      vat_number,
      tax_office,
      address,
      city,
      region,
      postal_code,
      country,
      phone1,
      phone2,
      phone3,
      phone4,
      email,
      iban,
      bank_name,
      contact_person,
      sales_station,
      collection_station,
      contract_date,
      credit_limit,
      area_manager,
      notes,
      is_active,
    } = data;

    await db.query(
      `INSERT INTO customers
      (code, name, trade_name, profession, vat_number, tax_office, address, city, region, postal_code, country,
      phone1, phone2, phone3, phone4, email, iban, bank_name, contact_person, sales_station, collection_station,
      contract_date, credit_limit, area_manager, notes, is_active)
      VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        code,
        name,
        trade_name,
        profession,
        vat_number,
        tax_office,
        address,
        city,
        region,
        postal_code,
        country,
        phone1,
        phone2,
        phone3,
        phone4,
        email,
        iban,
        bank_name,
        contact_person,
        sales_station,
        collection_station,
        contract_date || null,
        credit_limit || 0,
        area_manager,
        notes,
        is_active ?? 1,
      ]
    );

    res.status(201).json({ message: "✅ Πελάτης δημιουργήθηκε επιτυχώς." });
  } catch (err) {
    console.error("❌ Error creating customer:", err);
    res.status(500).json({ message: "Σφάλμα κατά τη δημιουργία πελάτη." });
  }
}

// === Ενημέρωση πελάτη ===
export async function updateCustomer(req, res) {
  try {
    const { id } = req.params;
    const data = req.body;

    await db.query(
      `UPDATE customers 
       SET code=?, name=?, trade_name=?, profession=?, vat_number=?, tax_office=?, address=?, city=?, region=?, postal_code=?, 
           country=?, phone1=?, phone2=?, phone3=?, phone4=?, email=?, iban=?, bank_name=?, contact_person=?, 
           sales_station=?, collection_station=?, contract_date=?, credit_limit=?, area_manager=?, notes=?, is_active=? 
       WHERE id=?`,
      [
        data.code,
        data.name,
        data.trade_name,
        data.profession,
        data.vat_number,
        data.tax_office,
        data.address,
        data.city,
        data.region,
        data.postal_code,
        data.country,
        data.phone1,
        data.phone2,
        data.phone3,
        data.phone4,
        data.email,
        data.iban,
        data.bank_name,
        data.contact_person,
        data.sales_station,
        data.collection_station,
        data.contract_date || null,
        data.credit_limit || 0,
        data.area_manager,
        data.notes,
        data.is_active ?? 1,
        id,
      ]
    );

    res.json({ message: "✅ Πελάτης ενημερώθηκε." });
  } catch (err) {
    console.error("❌ Error updating customer:", err);
    res.status(500).json({ message: "Σφάλμα κατά την ενημέρωση πελάτη." });
  }
}

// === Διαγραφή πελάτη ===
export async function deleteCustomer(req, res) {
  try {
    const { id } = req.params;
    await db.query("DELETE FROM customers WHERE id = ?", [id]);
    res.json({ message: "🗑️ Πελάτης διαγράφηκε." });
  } catch (err) {
    console.error("❌ Error deleting customer:", err);
    res.status(500).json({ message: "Σφάλμα κατά τη διαγραφή πελάτη." });
  }
}

// === Λήψη ενός πελάτη ===
export async function getCustomer(req, res) {
  try {
    const { id } = req.params;
    const [rows] = await db.query("SELECT * FROM customers WHERE id = ?", [id]);
    if (rows.length === 0) return res.status(404).json({ message: "Δεν βρέθηκε." });
    res.json(rows[0]);
  } catch (err) {
    console.error("❌ Error fetching customer:", err);
    res.status(500).json({ message: "Σφάλμα κατά τη λήψη πελάτη." });
  }
}

// === Import από Excel (ΕΛΤΑ) ===
// === Import από Excel (ΕΛΤΑ) ===
export async function importCustomers(req, res) {
  try {
    const workbook = XLSX.readFile(req.file.path);
    const sheet = workbook.Sheets[workbook.SheetNames[0]];
    const data = XLSX.utils.sheet_to_json(sheet);

    let inserted = 0, updated = 0, failed = 0;
    const details = { inserted: [], updated: [], failed: [] };

    for (const row of data) {
      try {
        const code = (row["CODES"] || "").trim();
        if (!code) continue;

        const parent_code = code.replace(/\s+/g, "").slice(0, 6);
        const name = (row["TITLES"] || "").trim();
        const vat_number = (row["AFM"] || "").trim();
        const trade_name = (row["SECOND_TITLE"] || "").trim();
        const profession = (row["TRADE_TITLE"] || "").trim();
        const email = row["EMAIL"] || row["EMAIL_1"] || null;
        const contact_person = row["CONDUCT_NAME"] || null;
        const phone1 = row["TEL_1"] || null;
        const phone2 = row["TEL_2"] || null;
        const phone3 = row["TEL_3"] || null;
        const phone4 = row["TEL_4"] || null;
        const address = `${row["ADDRES"] || ""} ${row["ADDRES_NUMBER"] || ""}`.trim();
        const city = row["AREASS"] || null;
        const region = row["COUNTY"] || null;
        const postal_code = row["TK"] || null;
        const tax_office = row["DOY"] || null;
        const bank_name = row["BANK"] || null;
        const iban = row["ACCOUNT"] || null;
        const sales_station = row["SALES"] || null;
        const collection_station = row["PAY_STATION"] || null;
        const area_manager = row["AREA_SALES"] || null;
        const credit_limit = row["CREDIT_LIMIT"] || 0;

        // === Ημερομηνία συμβολαίου (CON_REMMARKS) ===
        let contract_date = null;
        const rawContractValue = row["CON_REMMARKS"];

        if (rawContractValue && rawContractValue.toString().trim() !== "") {
          const raw = rawContractValue.toString().trim();

          if (/^\d{8}$/.test(raw)) {
            contract_date = `${raw.slice(0, 4)}-${raw.slice(4, 6)}-${raw.slice(6, 8)}`;
          } else if (!isNaN(raw) && Number(raw) > 10000) {
            const excelEpoch = new Date(Math.round((Number(raw) - 25569) * 86400 * 1000));
            contract_date = excelEpoch.toISOString().split("T")[0];
          } else if (raw.includes("/") || raw.includes(".")) {
            const cleaned = raw.replace(/\./g, "-").replace(/\//g, "-");
            const parts = cleaned.split("-");
            if (parts[0].length === 4) {
              contract_date = cleaned; // YYYY-MM-DD
            } else if (parts[2] && parts[2].length === 4) {
              contract_date = `${parts[2]}-${parts[1]}-${parts[0]}`;
            }
          }
        }

        // === Παρατηρήσεις (REMARKSS) ===
        let notes = null;
        const rawNotes = row["REMARKSS"];
        if (rawNotes && rawNotes.toString().trim() !== "") {
          notes = rawNotes.toString().trim();
        }

        // --- Έλεγχος ύπαρξης ---
        const [exists] = await db.query("SELECT id FROM customers WHERE code = ?", [code]);

        if (exists.length > 0) {
          // === UPDATE ===
          await db.query(
            `UPDATE customers SET
              parent_code=?, name=?, trade_name=?, profession=?, vat_number=?, email=?, phone1=?, phone2=?, phone3=?, phone4=?,
              address=?, city=?, region=?, postal_code=?, tax_office=?, contact_person=?, 
              bank_name=?, iban=?, sales_station=?, collection_station=?, 
              area_manager=?, contract_date=?, credit_limit=?, notes=? 
             WHERE id=?`,
            [
              parent_code, name, trade_name, profession, vat_number, email,
              phone1, phone2, phone3, phone4,
              address, city, region, postal_code, tax_office, contact_person,
              bank_name, iban, sales_station, collection_station,
              area_manager, contract_date, credit_limit, notes, exists[0].id,
            ]
          );
          updated++;
          details.updated.push({ code, name });
        } else {
          // === INSERT ===
          await db.query(
            `INSERT INTO customers
              (code, parent_code, name, trade_name, profession, vat_number, email, phone1, phone2, phone3, phone4, 
               address, city, region, postal_code, tax_office, contact_person, bank_name, iban, sales_station, 
               collection_station, area_manager, contract_date, credit_limit, notes)
             VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
            [
              code, parent_code, name, trade_name, profession, vat_number, email,
              phone1, phone2, phone3, phone4,
              address, city, region, postal_code, tax_office, contact_person,
              bank_name, iban, sales_station, collection_station,
              area_manager, contract_date, credit_limit, notes,
            ]
          );
          inserted++;
          details.inserted.push({ code, name });
        }
      } catch (innerErr) {
        failed++;
        details.failed.push({
          code: row["CODES"],
          name: row["TITLES"],
          error: innerErr.message,
        });
        console.error("⚠️ Import row error:", innerErr.message);
      }
    }

    fs.unlinkSync(req.file.path);
    res.json({ summary: { inserted, updated, failed }, details });
  } catch (err) {
    console.error("❌ Import error:", err);
    res.status(500).json({ message: "Σφάλμα κατά το import πελατών." });
  }
}
// === [ΝΕΟ] ΛΗΞΙΠΡΟΘΕΣΜΟΙ ΠΕΛΑΤΕΣ ===

export const getOverdueCustomers = async (req, res) => {
  try {
    const [rows] = await db.query(`
      SELECT 
        c.id AS customer_id,
        c.name AS customer_name,
        ROUND(SUM(i.amount) - COALESCE(SUM(p.amount), 0), 2) AS balance
      FROM customers c
      LEFT JOIN invoices i ON i.customer_id = c.id
      LEFT JOIN invoice_payments p ON p.customer_id = c.id
      GROUP BY c.id
      HAVING balance > 0
      ORDER BY balance DESC
    `);

    res.json(rows);
  } catch (err) {
    console.error("❌ Σφάλμα στο getOverdueCustomers:", err);
    res.status(500).json({ message: "Σφάλμα ανάκτησης ληξιπρόθεσμων πελατών" });
  }
};
