import pool from "../db.js";

// Δημιουργία νέας εγγραφής
export async function createCashSession(req, res) {
  const {
    user_id, branch_id, session_date, cash_type,
    system_cash, declared_cash, notes,
    pos_charges, pos_cod, pos_fastpay,
    cash_charges, cash_cod, cash_fastpay,
    denominations = [],
    expenses = [],
  } = req.body;

  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    const [result] = await conn.execute(
      `INSERT INTO cash_sessions (
        user_id, branch_id, session_date, cash_type,
        system_cash, declared_cash, notes,
        pos_charges, pos_cod, pos_fastpay,
        cash_charges, cash_cod, cash_fastpay
      ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
      [
        user_id, branch_id, session_date, cash_type,
        system_cash, declared_cash, notes,
        pos_charges, pos_cod, pos_fastpay,
        cash_charges, cash_cod, cash_fastpay,
      ]
    );

    const sessionId = result.insertId;

    for (const d of denominations) {
      await conn.execute(
        `INSERT INTO cash_denominations (session_id, value, quantity)
         VALUES (?, ?, ?)`,
        [sessionId, d.value, d.quantity]
      );
    }

    for (const exp of expenses) {
      await conn.execute(
        `INSERT INTO cash_expenses (session_id, amount, description, receipt_file)
         VALUES (?, ?, ?, ?)`,
        [sessionId, exp.amount, exp.description, exp.receipt_file || null]
      );
    }

    await conn.commit();
    res.status(201).json({ message: "Η καταμέτρηση καταχωρήθηκε", id: sessionId });
  } catch (err) {
    await conn.rollback();
    console.error("❌ createCashSession error:", err);
    res.status(500).json({ error: "Σφάλμα καταχώρησης" });
  } finally {
    conn.release();
  }
}

// Ανάκτηση όλων
export async function getCashSessions(req, res) {
  try {
    const [rows] = await pool.query(
      `SELECT cs.*, u.username, b.name AS branch_name
       FROM cash_sessions cs
       JOIN users u ON cs.user_id = u.id
       LEFT JOIN branches b ON cs.branch_id = b.id
       ORDER BY cs.session_date DESC, cs.id DESC`
    );

    const enriched = await Promise.all(rows.map(async (row) => {
      const [denoms] = await pool.query(
        `SELECT value, quantity FROM cash_denominations WHERE session_id = ?`,
        [row.id]
      );

      const [expenses] = await pool.query(
        `SELECT amount, description, receipt_file FROM cash_expenses WHERE session_id = ?`,
        [row.id]
      );

      return {
        ...row,
        user: { id: row.user_id, username: row.username },
        branch: { id: row.branch_id, name: row.branch_name },
        denominations: denoms,
        expenses,
      };
    }));

    res.json(enriched);
  } catch (err) {
    console.error("❌ getCashSessions error:", err);
    res.status(500).json({ error: "Σφάλμα ανάκτησης" });
  }
}

// Ανάκτηση μεμονωμένης
export async function getCashSessionById(req, res) {
  const { id } = req.params;
  try {
    const [sessionRows] = await pool.query(
      `SELECT cs.*, u.username, b.name AS branch_name
       FROM cash_sessions cs
       JOIN users u ON cs.user_id = u.id
       LEFT JOIN branches b ON cs.branch_id = b.id
       WHERE cs.id = ?`,
      [id]
    );

    if (!sessionRows.length) {
      return res.status(404).json({ error: "Δεν βρέθηκε" });
    }

    const session = sessionRows[0];

    const [denoms] = await pool.query(
      `SELECT value, quantity FROM cash_denominations WHERE session_id = ?`,
      [id]
    );

    const [expenses] = await pool.query(
      `SELECT amount, description, receipt_file FROM cash_expenses WHERE session_id = ?`,
      [id]
    );

    res.json({
      ...session,
      user: { id: session.user_id, username: session.username },
      branch: { id: session.branch_id, name: session.branch_name },
      denominations: denoms,
      expenses: expenses || [], // ✅ Αυτό φροντίζει να περάσει η λίστα
    });
  } catch (err) {
    console.error("❌ getCashSessionById error:", err);
    res.status(500).json({ error: "Σφάλμα ανάκτησης" });
  }
}

// Ενημέρωση
export async function updateCashSession(req, res) {
  const { id } = req.params;
  const {
    user_id, branch_id, session_date, cash_type,
    system_cash, declared_cash, notes,
    pos_charges, pos_cod, pos_fastpay,
    cash_charges, cash_cod, cash_fastpay,
    denominations = [],
    expenses = [],
  } = req.body;

  const conn = await pool.getConnection();
  try {
    await conn.beginTransaction();

    await conn.execute(
      `UPDATE cash_sessions
       SET user_id = ?, branch_id = ?, session_date = ?, cash_type = ?,
           system_cash = ?, declared_cash = ?, notes = ?,
           pos_charges = ?, pos_cod = ?, pos_fastpay = ?,
           cash_charges = ?, cash_cod = ?, cash_fastpay = ?
       WHERE id = ?`,
      [
        user_id, branch_id, session_date, cash_type,
        system_cash, declared_cash, notes,
        pos_charges, pos_cod, pos_fastpay,
        cash_charges, cash_cod, cash_fastpay,
        id,
      ]
    );

    await conn.execute(`DELETE FROM cash_denominations WHERE session_id = ?`, [id]);
    await conn.execute(`DELETE FROM cash_expenses WHERE session_id = ?`, [id]);

    for (const d of denominations) {
      await conn.execute(
        `INSERT INTO cash_denominations (session_id, value, quantity)
         VALUES (?, ?, ?)`,
        [id, d.value, d.quantity]
      );
    }

    for (const exp of expenses) {
      await conn.execute(
        `INSERT INTO cash_expenses (session_id, amount, description, receipt_file)
         VALUES (?, ?, ?, ?)`,
        [id, exp.amount, exp.description, exp.receipt_file || null]
      );
    }

    await conn.commit();
    res.json({ message: "Η καταμέτρηση ενημερώθηκε" });
  } catch (err) {
    await conn.rollback();
    console.error("❌ updateCashSession error:", err);
    res.status(500).json({ error: "Σφάλμα ενημέρωσης" });
  } finally {
    conn.release();
  }
}

// Διαγραφή
export async function deleteCashSession(req, res) {
  const { id } = req.params;
  try {
    await pool.query(`DELETE FROM cash_sessions WHERE id = ?`, [id]);
    res.json({ message: "Η καταγραφή διαγράφηκε" });
  } catch (err) {
    console.error("❌ deleteCashSession error:", err);
    res.status(500).json({ error: "Σφάλμα διαγραφής" });
  }
}
export async function uploadReceiptHandler(req, res) {
  const { sessionId } = req.params;
  const file = req.file;

  if (!file) return res.status(400).json({ message: "Δεν στάλθηκε αρχείο" });

  try {
    await pool.execute(
      `INSERT INTO cash_expenses (session_id, amount, description, receipt_file)
       VALUES (?, ?, ?, ?)`,
      [sessionId, 0, "Χειροκίνητο αρχείο", file.path]
    );
    res.status(201).json({ message: "Το παραστατικό ανέβηκε" });
  } catch (err) {
    console.error("❌ Receipt Upload Error:", err);
    res.status(500).json({ message: "Σφάλμα διακομιστή" });
  }
}