import pool from "../db.js";

// ✅ Get all branches
export async function getBranches(req, res) {
  try {
    const [rows] = await pool.query(
      "SELECT id, name, address, phone, created_at, branch_type FROM branches ORDER BY name ASC"
    );
    res.json(rows);
  } catch (err) {
    console.error("❌ Error fetching branches:", err);
    res.status(500).json({ message: "Server error" });
  }
}

// ✅ Get single branch
export async function getBranchById(req, res) {
  try {
    const { id } = req.params;
    const [[branch]] = await pool.query(
      "SELECT id, name, address, phone, created_at, branch_type FROM branches WHERE id = ?",
      [id]
    );
    if (!branch) return res.status(404).json({ message: "Branch not found" });
    res.json(branch);
  } catch (err) {
    console.error("❌ Error fetching branch:", err);
    res.status(500).json({ message: "Server error" });
  }
}

// ✅ Create branch
export async function createBranch(req, res) {
  try {
    const { name, address, phone, branch_type } = req.body;
    if (!name)
      return res.status(400).json({ message: "Name is required" });

    const [result] = await pool.query(
      "INSERT INTO branches (name, address, phone, branch_type) VALUES (?, ?, ?, ?)",
      [name, address, phone, branch_type]
    );

    res.status(201).json({
      id: result.insertId,
      name,
      address,
      phone,
      branch_type,
    });
  } catch (err) {
    console.error("❌ Error creating branch:", err);
    res.status(500).json({ message: "Server error" });
  }
}

// ✅ Update branch
export async function updateBranch(req, res) {
  try {
    const { id } = req.params;
    const { name, address, phone, branch_type } = req.body;

    const [result] = await pool.query(
      "UPDATE branches SET name=?, address=?, phone=?, branch_type=? WHERE id=?",
      [name, address, phone, branch_type, id]
    );

    if (result.affectedRows === 0)
      return res.status(404).json({ message: "Branch not found" });

    res.json({ id, name, address, phone, branch_type });
  } catch (err) {
    console.error("❌ Error updating branch:", err);
    res.status(500).json({ message: "Server error" });
  }
}

// ✅ Delete branch
export async function deleteBranch(req, res) {
  try {
    const { id } = req.params;
    const [result] = await pool.query("DELETE FROM branches WHERE id = ?", [id]);

    if (result.affectedRows === 0)
      return res.status(404).json({ message: "Branch not found" });

    res.json({ message: "Branch deleted successfully" });
  } catch (err) {
    console.error("❌ Error deleting branch:", err);
    res.status(500).json({ message: "Server error" });
  }
}
